package com.franz.ag;

import java.util.NoSuchElementException;

import com.franz.ag.AllegroGraphException;
import com.franz.ag.Cursor;
import com.franz.ag.UPI;
import com.franz.ag.impl.CursorImpl;
import com.hp.hpl.jena.datatypes.RDFDatatype;
import com.hp.hpl.jena.datatypes.TypeMapper;
import com.hp.hpl.jena.graph.Node;
import com.hp.hpl.jena.graph.Triple;
import com.hp.hpl.jena.graph.TripleIterator;
import com.hp.hpl.jena.util.iterator.NiceIterator;

/**
 * Implement ExtendedIterator from AllegroGraph Cursor.
 * 
 * @author mm
 *
 */
public class JenaCursor extends NiceIterator implements TripleIterator {

	private CursorImpl agcc;
	private AllegroGraphJenaImpl agji;
	
	JenaCursor ( AllegroGraphJenaImpl aj, Cursor cc ) {
		super();
		agcc = (CursorImpl) cc;  agji = aj;
	}

	public void close() {
		agcc.close();
	}

	public boolean hasNext() {
		return agcc.hasNext();
	}

	/**
	 * Create a Jena Node instance from AllegroGraph parts.
	 * 
	 * @param id The node id number
	 * @param type The node type
	 * @param val The node label
	 * @param mod The node language or datatype modifier
	 * @return a Jena Node instance
	 */
	Node encodeNode( UPI id, int type, String val, String mod ) {
		Node r = encodeNodeInner(id, type, val, mod);
		if ( ifDebug(2) )
		{
			try {
				agcc.getAG().ags.agc.evalInServer("(format t \"~&encodeNode: " + id + "  " + type + "  " + 
									val + "  " + mod + 
									" ==> " + r.getClass().getName() + "~%\")"
									);
			} catch (AllegroGraphException e) {	}
		}
		
		
		return r;
	}
	
	
	Node encodeNodeInner( UPI id, int type, String val, String mod ) {
		// types:  1     2     3        4             5
		//        anon  node  literal  literal/lang  typed-literal
		if ( type==1 )
			return agji.lookupAGanon(id, true);
		else if ( type==2 )
				return Node.createURI(val);
		else if ( type==3 )
				return Node.createLiteral(val);
		else if ( type==4 )
				return Node.createLiteral(val, mod, false);	
		else if ( type==5 )
		{
			// Code suggested by Holger 28 Nov 2006
			RDFDatatype datatype = TypeMapper.getInstance().getTypeByName(mod);
			return Node.createLiteral(val, null, datatype);
				//return Node.createLiteral(val, "", new BaseDatatype(mod));
		}
		else throw new IllegalArgumentException("Cannot encode AG " 
				+ type + "=" + agcc.getAG().typeToString(type) +
				"/" + val + "/" + mod + " to Jena Node");
	}
	
	boolean ifDebug( int index ) {
		return agcc.getAG().ags.ifDebug(index);
	}
	
	public Object next() {
		Object v = null;
		if ( ifDebug( 1 ) )
		{	
			try {
				agcc.getAG().ags.serverTrace(true);
				v = nextInternal();
			} finally { agcc.getAG().ags.serverTrace(false); }
		}
		else
			v = nextInternal();
		return v;
	}
	
	public Object nextInternal() {
		try {
			if ( agcc.step() ) 
			{
				Triple tr = new Triple(
						encodeNode(agcc.getS(), agcc.baseInstance.getPartType(1),
								agcc.baseInstance.getPartLabel(1), agcc.baseInstance.getPartMod(1)),
						encodeNode(agcc.baseInstance.getP(), agcc.baseInstance.getPartType(2),
								agcc.baseInstance.getPartLabel(2), agcc.baseInstance.getPartMod(2)),
						encodeNode(agcc.getO(), agcc.baseInstance.getPartType(3),
								agcc.baseInstance.getPartLabel(3), agcc.baseInstance.getPartMod(3))
						);
				return tr;
				}
			throw new NoSuchElementException();
		} catch (AllegroGraphException e) {
			throw new IllegalStateException(e.toString());
		}	
	}
	
	

	public void remove() {
		agcc.remove();
	}

	public Triple nextTriple() {
		return (Triple)next();
	}

}
