package com.franz.agbase.examples;

import com.franz.agbase.*;

public class AGGeospatialCartesianData {

	/**
	 * Demonstrates some basics of working with Cartesian data.
	 * 
	 * @param args unused
	 * @throws AllegroGraphException
	 */
	public static void main(String[] args) throws AllegroGraphException {
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}

		// Create a fresh triple-store.
		AllegroGraph ts = ags.renew("geospatialcartesiandata", AGPaths.TRIPLE_STORES);
		
		// Register namespaces
		ts.registerNamespace("ex","http://example.org/places/");
		ts.registerNamespace("geo","http://franz.com/ns/allegrograph/3.0/geospatial/");
		
		// Get a GeoExtension instance for this store to work with geospatial features 
		GeoExtension geo = ts.getGeoExtension();
		
		// Define a geospatial subtype to encode coordinates 
		GeospatialSubtype subtype = geo.registerCartesianStriping(0, 1, 0, 1, .01);
		geo.addSubtype(subtype);
		
		// Encode the coordinates as a Geospatial UPI
		// Note the argument ordering is x, y
		UPI upi = geo.encodeUPI(subtype, .25, .75);

		// Decode the coordinates from the UPI
		Object[] location = geo.decodeUPI(upi);
		AGUtils.printObjectArray("Location:", location);
		
		// Add triples to the store
		ts.addStatement("!ex:pointA", "!ex:location", upi);
		
		// Show all triples in the store
		TriplesIterator cc = ts.getStatements(null,null,null);
		AGUtils.showTriples(cc);
		
		// Serialize the store to an NTriples file
		NTriplesSerializer ntser = new NTriplesSerializer();
		ntser.setDestination(AGPaths.dataSources("cartesian-serialized.nt"));
		ntser.setIfExists("supersede");
		ntser.run(ts);
		
		// Close the store and disconnect from the server.
		ts.closeTripleStore();
		ags.disable();
	}

}
