package com.franz.agbase.examples;

import com.franz.agbase.*;

public class AGIndexAllTriples {

	/**
	 * Demonstrates loading and querying of the Wilburwine dataset.
	 * 
	 * @param args unused
	 * @throws AllegroGraphException
	 */
	public static void main(String[] args) throws AllegroGraphException {
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}

		// Create a fresh triple store
		AllegroGraph ts = ags.renew("wilburwine", AGPaths.TRIPLE_STORES);

		// Load the wilburwine dataset
		String ntripleFile = AGPaths.dataSources("wilburwine.ntriples");		
		System.out.println("Loading N-Triples " + ntripleFile);
		AGLoadNtriples.loadNTriplesWithTiming(ts,ntripleFile);
		
		// Register a wine namespace
		ts.registerNamespace("wine", "http://www.w3.org/TR/2003/CR-owl-guide-20030818/wine#");
		
		// Get triples having subject CabernetSauvignonGrape and show them
		System.out.println("Triples with Subject: CabernetSauvignonGrape");
		long t0 = System.nanoTime();
		TriplesIterator cc = ts.getStatements("!wine:CabernetSauvignonGrape",null, null);
		AGUtils.showTriples(cc);
		long t1 = System.nanoTime();
		System.out.println("Query (before indexing) took " + (long)(t1-t0) + " nanoseconds");

		// Index the triple store for faster querying
		ts.indexAllTriples();

		// Get triples having subject CabernetSauvignonGrape and show them
		System.out.println("Triples with Subject: CabernetSauvignonGrape");
		long t2 = System.nanoTime();
		cc = ts.getStatements("!wine:CabernetSauvignonGrape",null, null);
		AGUtils.showTriples(cc);
		long t3 = System.nanoTime();
		System.out.println("Query (after indexing) took " + (long)(t3-t2) + " nanoseconds");

		// Close the triple store and disconnect from the server.
		ts.closeTripleStore();
		ags.disable();
	}

	public static void indexAllTriplesWithTiming(AllegroGraph ts) throws AllegroGraphException {
		System.out.print("Indexing...");
		long start = System.currentTimeMillis();
		ts.indexAllTriples(true);
		System.out.println("done in " + AGUtils.elapsedTime(start));
	}

}
