package com.franz.agbase.examples;

import com.franz.agbase.*;

public class AGLoadNtriples {

	/**
	 * Demonstrates how to load a triple store from N-Triples files.
	 * 
	 * @param args unused
	 * @throws AllegroGraphException
	 */
	public static void main(String[] args) throws AllegroGraphException {
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}

		// Create fresh triple-store.
		AllegroGraph ts = ags.renew("loadntriples", AGPaths.TRIPLE_STORES);
		
		// Load a file in N-Triples format
		String ntripleFile = AGPaths.dataSources("test.nt");		
		System.out.println("Loading N-Triples " + ntripleFile);
		ts.loadNTriples(ntripleFile);
		System.out.println("Loaded " + ts.numberOfTriples() + " triples.");
		
		// Get all triples from the default graph and show them.
		TriplesIterator it = ts.getStatements(null, null, null);
		AGUtils.showTriples(it);
		
		// Load N-Triples from a file into a named graph
		URINode g = ts.createURI("http://example.org/kennedy");
		long n = ts.loadNTriples(AGPaths.dataSources("kennedy.ntriples"), g);
		System.out.println("Loaded " + n + " triples.");
		
		// Get some triples from a given graph g and show them.
		it = ts.getStatements(null, null, ts.createLiteral("Arnold"), g);
		AGUtils.showTriples(it);
		
		// Load N-Triples from a URL into a named graph
		loadNTriplesWithTiming(ts,"http://www.w3.org/2000/10/rdf-tests/rdfcore/ntriples/test.nt", "source");
		
		// Get some triples and show them, noting the graph.
		it = ts.getStatements("<http://example.org/resource1>", null, null, null);
		AGUtils.showTriples(it);
		
		// If you have a number of files to load, it is more efficient to
		// provide an array of filenames rather than loading them separately.
		String[] files = { 
				AGPaths.dataSources("sna.nt"),
				AGPaths.dataSources("temporal.nt"),
				AGPaths.dataSources("wilburwine.ntriples")
				,"http://www.w3.org/2000/10/rdf-tests/rdfcore/ntriples/test.nt"		
				};
		loadNTriplesWithTiming(ts,files,"source");
		
		// Get some triples and note their source graph.
		it = ts.getStatements("<http://example.org/a>", null, null, null);
		AGUtils.showTriples(it);

		// Close the store and disconnect from the server.
		ts.closeTripleStore();
		ags.disable();
	}

	/**
	 * Load a single NTriples file into the default graph and time the load.
	 * 
	 * @param ts A triple store 
	 * @param ntriplesFile A server-accessible NTriples file 
	 * @throws AllegroGraphException
	 */
	public static void loadNTriplesWithTiming(AllegroGraph ts, String ntriplesFile) throws AllegroGraphException {
		AGLoadNtriples.loadNTriplesWithTiming(ts, ntriplesFile, "");
	}

	/**
	 * Load a single NTriples file into the specified graph and time the load.
	 * 
	 * @param ts A triple store 
	 * @param ntriplesFile A server-accessible NTriples file
	 * @param graph The context to load 
	 * @throws AllegroGraphException
	 */
	public static void loadNTriplesWithTiming(AllegroGraph ts, String ntriplesFile, String graph) throws AllegroGraphException {
		System.out.println("Loading N-Triples from " + ntriplesFile);
		long start = System.currentTimeMillis();
		long n = ts.loadNTriples(ntriplesFile,graph,false,null,null);
		System.out.println("Loaded " + n + " triples in " + AGUtils.elapsedTime(start));
	}
	
	/**
	 * Load an array of NTriples files into the specified graph and time the load.
	 * 
	 * @param ts A triple store 
	 * @param ntriplesFiles An array of server-accessible NTriples files
	 * @param graph The context to load 
	 * @return
	 * @throws AllegroGraphException
	 */
	public static long loadNTriplesWithTiming(AllegroGraph ts, String[] ntriplesFiles, String graph) throws AllegroGraphException {
		System.out.println("Loading " + ntriplesFiles.length + " N-Triples files.");
		long start = System.currentTimeMillis();
		long n = ts.loadNTriples(ntriplesFiles,graph,false,null,null);
		System.out.println("Loaded " + n + " triples in " + AGUtils.elapsedTime(start));
		return n;
	}

}
