package com.franz.agbase.examples;

import com.franz.agbase.AllegroGraph;
import com.franz.agbase.AllegroGraphConnection;
import com.franz.agbase.AllegroGraphException;

public class AGLubm50Build {

	/**
	 * Demonstrates loading and indexing a triple store with the LUBM 50 
	 * benchmark data.  This data includes the minimal set of additional 
	 * triples (1000 triples) inferred ahead of time using OWL (non-RDFS++) 
	 * reasoning that reduces the LUBM queries to RDFS++ challenge problems.  
	 * Please refer to the files AGLubm50Prolog.java and AGLubm50SPARQL.java 
	 * to run these queries using Prolog and SPARQL-RDFS++ reasoning, resp.
	 * 
	 * The data that is necessary to run this example is available at:
	 * http://franzdownload.com/allegrograph/data/lubm50rdfs++.nt.bz2
	 * 
	 * The file is about 1.2GB when unpacked.  The example assumes that
	 * the file will be unpacked in the AGPaths.DATA_SOURCES directory; 
	 * modify this location as necessary.
	 * 
	 * @throws AllegroGraphException 
	 */		
	public static void main(String[] args) throws AllegroGraphException {

		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.setPort(4126);
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}
		
		// Estimate the number of unique resources in the Lubm 50 data
		// Simple heuristic: a little more than 1/3 of the total triples  
		ags.setDefaultExpectedResources(2500000);

		// Create fresh triple-store for this example.
		AllegroGraph ts = ags.renew("Lubm-50", AGPaths.TRIPLE_STORES);

		// Load the N-Triples file -- modify the location as necessary.
		AGLoadNtriples.loadNTriplesWithTiming(ts, AGPaths.dataSources("lubm50rdfs++.nt"));

		// Index the store -- save disk space by only using 3 indices
		ts.setIndexFlavors(new String[]{"spogi","posgi","ospgi"});
		AGIndexAllTriples.indexAllTriplesWithTiming(ts);

		// Close the store and disconnect from the server
		ts.closeTripleStore();
		ags.disable();
 	}

}
