package com.franz.ag.repository.examples;

import static com.franz.ag.repository.examples.AGRepositoryValueFactory.*;

import java.io.FileInputStream;
import java.io.InputStream;
import java.net.URL;

import com.franz.ag.repository.AGRepository;
import com.franz.agbase.AllegroGraph;
import com.franz.agbase.AllegroGraphConnection;
import com.franz.agbase.AllegroGraphException;
import com.franz.agbase.examples.AGPaths;

import org.openrdf.model.Statement;
import org.openrdf.model.ValueFactory;
import org.openrdf.repository.Repository;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.rio.RDFFormat;

public class AGRepositoryAdd {

	/**
	 * Demonstrates various ways of adding data to a repository. 
	 * 
	 * @param args unused
	 * @throws AllegroGraphException
	 */
	public static void main(String[] args) throws Exception {
		// Connect to the server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}

		// Get a connection to a fresh repository
		AllegroGraph ts = ags.renew("repositoryadd", AGPaths.TRIPLE_STORES);
		Repository repo = new AGRepository(ts);
		repo.initialize();
		RepositoryConnection repoConn = repo.getConnection();
		
		// Load RDFXML from a URL into the null context
		repoConn.add(new URL("http://www.w3.org/TR/owl-guide/wine.rdf"), "", RDFFormat.RDFXML);
		System.out.println("Loaded " + repoConn.size() + " triples.");
		
		// Get the ValueFactory for the Repository and create some example values. 
		ValueFactory vf = repo.getValueFactory();
		createExampleValues(vf);

		repoConn.add(bob, name, nameBob);
		System.out.println("Check for added statement: " + 
				repoConn.hasStatement(bob, name, nameBob, false));
		
		Statement statement = vf.createStatement(alice, name, nameAlice);
		repoConn.add(statement);

		System.out.println("Check for added statement: " +
				repoConn.hasStatement(statement, false));
		System.out.println("Check for added statement: " +
				repoConn.hasStatement(alice, name, nameAlice, false));

		// Load Ntriples from a local file
		InputStream defaultGraph = new FileInputStream(AGPaths.dataSources("default-graph.nt"));
		try {
			repoConn.add(defaultGraph, "", RDFFormat.NTRIPLES);
		} finally {
			defaultGraph.close();
		}

		System.out.println("Repository contains newly added statements: " +
				repoConn.hasStatement(null, publisher, nameBob, false));
		System.out.println("Repository contains newly added statements: " +
				repoConn.hasStatement(null, publisher, nameAlice, false));

		// add file graph1.nt to context1
		InputStream graph1 = new FileInputStream(AGPaths.dataSources("graph1.nt"));
		try {
			repoConn.add(graph1, "", RDFFormat.NTRIPLES, context1);
		} finally {
			graph1.close();
		}

		// add file graph2.nt to context2
		InputStream graph2 = new FileInputStream(AGPaths.dataSources("graph2.nt"));
		try {
			repoConn.add(graph2, "", RDFFormat.NTRIPLES, context2);
		} finally {
			graph2.close();
		}

		System.out.println("alice is known in context1?: " +
				repoConn.hasStatement(null, name, nameAlice, false, context1));
		System.out.println("alice is known in context2?: " +
				repoConn.hasStatement(null, name, nameAlice, false, context2));

		System.out.println("bob is known in context2?: " +
				repoConn.hasStatement(null, name, nameBob, false, context2));
		System.out.println("bob is known in context1?: " +
				repoConn.hasStatement(null, name, nameBob, false, context1));

		// Shutdown the Repository, close the store and disconnect from the server
		repoConn.close();
		repo.shutDown();
		ts.closeTripleStore();
		ags.disable();
	}
}
