package com.franz.ag.repository.examples;

import static com.franz.ag.repository.examples.AGRepositoryValueFactory.*;
import info.aduna.iteration.CloseableIteration;

import org.openrdf.model.Statement;
import org.openrdf.model.ValueFactory;
import org.openrdf.repository.Repository;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.repository.RepositoryResult;

import com.franz.ag.repository.AGRepository;
import com.franz.agbase.AllegroGraph;
import com.franz.agbase.AllegroGraphConnection;
import com.franz.agbase.AllegroGraphException;
import com.franz.agbase.examples.AGPaths;

public class AGRepositoryGetStatements {

	/**
	 * Demonstrates basics of using getStatements 
	 * 
	 * @param args unused
	 * @throws AllegroGraphException
	 */
	public static void main(String[] args) throws Exception {
		// Connect to the server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}

		// Get a connection to a fresh repository
		AllegroGraph ts = ags.renew("repositorygetstatements", AGPaths.TRIPLE_STORES);
		Repository repo = new AGRepository(ts);
		repo.initialize();
		RepositoryConnection repoConn = repo.getConnection();
		
		// Get the ValueFactory for the Repository and create some example values. 
		ValueFactory vf = repo.getValueFactory();
		createExampleValues(vf);

		repoConn.add(bob, name, nameBob);

		RepositoryResult<Statement> result = repoConn.getStatements(null, name,
				null, false);

		try {
			while (result.hasNext()) {
				Statement st = result.next();
				System.out.print(st.getSubject());
				System.out.print(", " + st.getPredicate());
				System.out.print(", " + st.getObject().stringValue());
				System.out.println(", " + st.getContext());
			}
		} finally {
			result.close();
		}

		repoConn.add(bob, name, nameBob, context1);
		repoConn.add(bob, mbox, mboxBob, context1);
		repoConn.add(context1, publisher, nameBob);

		repoConn.add(alice, name, nameAlice, context2);
		repoConn.add(alice, mbox, mboxAlice, context2);
		repoConn.add(context2, publisher, nameAlice);

		System.out.println("Repository contains statement: " +
				repoConn.hasStatement(bob, name, nameBob, false));

		System.out.println("Repository contains statement in context1: " +
				repoConn.hasStatement(bob, name, nameBob, false, context1));

		System.out.println("Repository contains statement in context2: " +
				repoConn.hasStatement(bob, name, nameBob, false, context2));

		System.out.println("Check handling of getStatements without context IDs:");
		result = repoConn.getStatements(bob, name, null, false);
		try {
			while (result.hasNext()) {
				Statement st = result.next();
				System.out.print(st.getSubject());
				System.out.print(", " + st.getPredicate());
				System.out.print(", " + st.getObject().stringValue());
				System.out.println(", " + st.getContext());
			}
		} finally {
			result.close();
		}

		System.out.println("Check handling of getStatements with a known context ID:");
		result = repoConn.getStatements(null, null, null, false, context1);
		try {
			while (result.hasNext()) {
				Statement st = result.next();
				System.out.print(st.getSubject());
				System.out.print(", " + st.getPredicate());
				System.out.print(", " + st.getObject().stringValue());
				System.out.println(", " + st.getContext());
			}
		} finally {
			result.close();
		}
		
		System.out.println("Get statements with either no context or context2:");
		CloseableIteration<? extends Statement, RepositoryException> iter = repoConn
				.getStatements(null, null, null, false, null, context2);

		try {
			while (iter.hasNext()) {
				Statement st = iter.next();
				System.out.print(st.getSubject());
				System.out.print(", " + st.getPredicate());
				System.out.print(", " + st.getObject().stringValue());
				System.out.println(", " + st.getContext());
			}
		} finally {
			iter.close();
		}

		System.out.println("Get all statements with context1 or context2:");
		iter = repoConn.getStatements(null, null, null, false, context1,
				context2);

		try {
			while (iter.hasNext()) {
				Statement st = iter.next();
				System.out.print(st.getSubject());
				System.out.print(", " + st.getPredicate());
				System.out.print(", " + st.getObject().stringValue());
				System.out.println(", " + st.getContext());
			}
		} finally {
			iter.close();
		}
		
		// Close the RepositoryConnection and shutdown the Repository
		// Close the store and disconnect from the server
		repoConn.close();
		repo.shutDown();
		ts.closeTripleStore();
		ags.disable();
	}
}
