package com.franz.agbase.examples;

import com.franz.agbase.*;

public class AGEvalInServer {
	/**
	 * Demonstrates some basics of evaluating lisp in the server.
	 * 
	 * To run this and other examples that use evalInServer, you will need
	 * to start the server using the --eval-in-server-file option, as in:
	 * 
	 * $ ./AllegroGraphServer --eval-in-server-file evalinserver.cfg
	 * 
	 * Refer also to the Learning Center under Starting a Server Manually
	 * for more information.
	 * 
	 * Note that this is an advanced feature for users that understand the
	 * internals of the AllegroGraph server.  The purpose of this example
	 * is simply to convey that one can indeed command the server in Lisp 
	 * through the Java API.
	 *   
	 * @param args unused
	 * @throws AllegroGraphException 
	 */
	public static void main(String[] args) throws AllegroGraphException {
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}
		
		// Add two numbers in the server and show the result
		evalInServer("(+ 2 3)",ags);
		
		// Define and call a function to add two numbers
		evalInServer("(defun foo (x y) (+ x y))",ags);
		evalInServer("(foo 4 5)",ags);
		
		// You can send a sequence of commands in a single call
		evalInServer("(progn (defun bar (x y) (+ x y)) (bar 6 7))",ags);
		
		// Symbols come back from the server as qualified names
		evalInServer("'a",ags);
		evalInServer("t",ags);

		// Disconnect from the server
		ags.disable();
	}

	public static void evalInServer(String lispexpr, AllegroGraphConnection ags) throws AllegroGraphException {
		evalInServer(lispexpr, true, ags);
	}
	
	public static void evalInServer(String lispexpr, boolean showResult, AllegroGraphConnection ags) throws AllegroGraphException {
		Object[] result;
		try {
			result = ags.evalInServer(lispexpr);
			if (showResult) AGUtils.printObjectArray("Server> "+ lispexpr, result);
		} catch (IllegalArgumentException e) {
			if (e.getMessage().contains("eval not permitted")) {
				throw new AllegroGraphException("evalInServer not permitted: Please ensure you have started the server with the --eval-in-server-file option.");
			} else {
				throw new AllegroGraphException(e.getMessage());
			}
		}
	}
	
}
