package com.franz.agbase.examples;

import com.franz.agbase.*;

public class AGFederationOfFederations {

	/**
	 * Demonstrates federating stores that are also federations.
	 * 
	 * @param args unused
	 * @throws AllegroGraphException
	 */
	public static void main(String[] args) throws AllegroGraphException {
		
		// Connect to the default server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem.", e);
		}

		// Register namespaces
		// Future AllegroGraph instances on this connection will get this.
		ags.registerNamespace("ex", "http://example.org/");
		
		// Create triple stores for this example
		AllegroGraph ts1 = ags.renew("store1", AGPaths.TRIPLE_STORES);
		AllegroGraph ts2 = ags.renew("store2", AGPaths.TRIPLE_STORES);
		AllegroGraph ts3 = ags.renew("store3", AGPaths.TRIPLE_STORES);
		
		// Create a federated store consisting of the first 2 open stores above 
		AllegroGraph[] parts1 = {ts1,ts2};
		AllegroGraph fed1 = ags.federate("federation1", parts1, true);
		
		// Create a federated store consisting of fed1 and the third store 
		AllegroGraph[] parts2 = {fed1,ts3};
		AllegroGraph fed2 = ags.federate("federation2", parts2, true);

		// Show stores in each federation 
		AllegroGraph[] stores1 = fed1.getStores();
		AGUtils.printObjectArray("federation1 stores:", stores1);
		AllegroGraph[] stores2 = fed2.getStores();
		AGUtils.printObjectArray("federation2 stores:", stores2);

		// Add some triples to the default graph of each store
		ts1.addStatement("!ex:A", "!rdfs:subClassOf", "!ex:B");
		ts2.addStatement("!ex:B", "!rdfs:subClassOf", "!ex:C");
		ts3.addStatement("!ex:a", "!rdf:type", "!ex:A");
		
		// Show all triples in the default graph of store 1.
		TriplesIterator cc = ts1.getStatements(null, null, null);
		System.out.println("Showing all triples in the default graph of store1:");
		AGUtils.showTriples(cc);
		
		// Show all triples in the default graph of store 2.
		cc = ts2.getStatements(null, null, null);
		System.out.println("Showing all triples in the default graph of store2:");
		AGUtils.showTriples(cc);
		
		// Show all triples in the default graph of store 3.
		cc = ts3.getStatements(null, null, null);
		System.out.println("Showing all triples in the default graph of store3:");
		AGUtils.showTriples(cc);
		
		// Retrieve all triples in fed2.
		// Note that we're using a wildcard on the graph argument.
		cc = fed2.getStatements(null, null, null, null);
		System.out.println("Retrieving all triples in federation2:");
		AGUtils.showTriples(cc);
		
		// Show all triples inferred from the federation.
		cc = fed2.getStatements(true, null, null, null, null);
		System.out.println("Showing all triples inferred from federation2:");
		AGUtils.showTriples(cc);
		
		// Close all stores and disconnect from the server
		fed2.closeTripleStore();
		fed1.closeTripleStore();
		ts1.closeTripleStore();
		ts2.closeTripleStore();
		ags.disable();
	}
	
}
