package com.franz.agbase.examples;

import com.franz.agbase.*;

public class AGFreetextSearch {

	/**
	 * Demonstrates some basics of Freetext search 
	 *
	 * @param args unused
	 * @throws AllegroGraphException
	 */
	public static void main(String[] args) throws AllegroGraphException {		
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}

		// Create a fresh triple store
		AllegroGraph ts = ags.renew("freetextsearch", AGPaths.TRIPLE_STORES);

		// Register a namespace
		ts.registerNamespace("ex", "http://example.org/");
		
		// Register any predicates whose objects are to be searched.
		// Only triples added or loaded after the registration will
		// be included in the index.
		ts.registerFreetextPredicate("<http://www.w3.org/2000/01/rdf-schema#comment>");
		ts.registerFreetextPredicate("!ex:notes");
		
		// Add some triples to the store.  Triples involving registered predicates will be 
		// indexed for free-text search, others will not.
		ts.addStatement("!ex:jans", "!rdfs:comment", "\"Born in Emmen in the Netherlands\"");
		ts.addStatement("!ex:gary", "!rdfs:comment", "\"Born in Springfield in the USA\"");
		ts.addStatement("!ex:henk", "!rdfs:label", "\"Born in Emmermeer in the Netherlands\"");
		ts.addStatement("!ex:alice", "!ex:notes", "\"wasn't born in Wonderland.\"");
		ts.addStatement("!ex:bob", "!ex:notes", "\"your uncle.\"");
		ts.addStatement("!ex:fred", "!ex:notes", "\"Lives in Bedrock.\"");
		ts.addStatement("!ex:stanley", "!ex:notes", "\"Dr. Livingstone I presume?\"");
		ts.addStatement("!ex:fibonacci", "!ex:notes", "\"fib(n+2) = fib(n+1) + fib(n)\"");
		
		// Search for statements whose object contains the word born -- note that
		// search is case insensitive
		String query = "'born'";
		System.out.println("Search: " + query);
		TriplesIterator cc = ts.getFreetextStatements(query);
		AGUtils.showTriples(cc);
		
		// Search for statements whose object contains words emmen and born
		query = "(and 'emmen' 'born')";
		System.out.println("Search: " + query);
		cc = ts.getFreetextStatements(query);
		AGUtils.showTriples(cc);
		
		// Search for statements whose object contains words Wonderland or Bedrock
		query = "(or 'Wonderland' 'Bedrock')";
		System.out.println("Search: " + query);
		cc = ts.getFreetextStatements(query);
		AGUtils.showTriples(cc);
		
		// Words of 2 chars or less are not indexed
		query = "(or 'in' 'a')";
		System.out.println("Search: " + query);
		cc = ts.getFreetextStatements(query);
		AGUtils.showTriples(cc);
		
		// Some common words, called "stop words", are also not indexed
		query = "'the'";
		System.out.println("Search: " + query);
		cc = ts.getFreetextStatements(query);
		AGUtils.showTriples(cc);
		
		// Some characters, such as '(' and ')' delimit tokens.
		// Note: 'fib(x)' will tokenize as "fib" and "x", and "x"
		// will be ignored having < 2 letters, so it's really a 
		// search for 'fib'.  Try searching for 'fib(xxx)' and 
		// note the different results.
		query = "'fib(x)'";
		System.out.println("Search: " + query);
		cc = ts.getFreetextStatements(query);
		AGUtils.showTriples(cc);
		
		// Match a single character with '?'
		query = "'Liv??'";
		System.out.println("Search: " + query);
		cc = ts.getFreetextStatements(query);
		AGUtils.showTriples(cc);
		
		// Match a sequence of characters with '*'
		query = "'Liv*'";
		System.out.println("Search: " + query);
		cc = ts.getFreetextStatements(query);
		AGUtils.showTriples(cc);
		
		// Search for subjects whose object contains Wonderland or Bedrock
		query = "(or 'Wonderland' 'Bedrock')";
		ValueSetIterator it = ts.getFreetextUniqueSubjects(query);
		System.out.println("Subjects matching search: " + query);
		AGUtils.showResults(it);
		
		// Close the triple store and disconnect from the server
		ts.closeTripleStore();
		ags.disable();
	}

}
