package com.franz.agbase.examples;

import com.franz.agbase.*;

public class AGGeospatialLoaderSerializer {

	/**
	 * Demonstrates loading and serializing Geospatial data.
	 * 
	 * @param args unused
	 * @throws AllegroGraphException
	 */
	public static void main(String[] args) throws AllegroGraphException {
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}

		// Create a fresh triple-store.
		AllegroGraph ts = ags.renew("geospatialloadserialize", AGPaths.TRIPLE_STORES);
		
		// Register namespaces
		ts.registerNamespace("ex","http://example.org/places/");
		ts.registerNamespace("geo","http://franz.com/ns/allegrograph/3.0/geospatial/");
		
		// Get a GeoExtension instance for this store to work with geospatial features 
		GeoExtension geo = ts.getGeoExtension();
		
		// Define a geospatial subtype to encode coordinates 
		GeospatialSubtype subtype50 = geo.registerLatitudeStripingInMiles(50.0);
		geo.addSubtype(subtype50);
		
		// Encode the coordinates as a Geospatial UPI
		// Note the argument ordering is x-longitude, y-latitude
		UPI upi = geo.encodeUPI(subtype50, -122.31307, 37.86385);

		// Decode the coordinates from the UPI
		Object[] location = geo.decodeUPI(upi);
		AGUtils.printObjectArray("Location:", location);
		
		// Add triples to the store
		ts.addStatement("!ex:Berkeley", "!ex:location", upi);
		
		// Show all triples in the store
		// Note that the location is printed in ISO 6709 format wherein
		// the ordering is latitude-longitude.
		TriplesIterator cc = ts.getStatements(null,null,null);
		AGUtils.showTriples(cc);
		
		// Serialize the store to an NTriples file
		NTriplesSerializer ntser = new NTriplesSerializer();
		ntser.setDestination(AGPaths.dataSources("geospatial-serialized.nt"));
		ntser.setIfExists("supersede");
		ntser.run(ts);
		
		// Close the store and disconnect from the server.
		ts.closeTripleStore();
		ags.disable();
	}

}
