package com.franz.agbase.examples;

import com.franz.agbase.*;

public class AGGeospatialPolygons {

	/**
	 * Demonstrates some basics of working with Polygons.
	 * 
	 * @param args unused
	 * @throws AllegroGraphException
	 */
	public static void main(String[] args) throws AllegroGraphException {
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}

		// Create a fresh triple-store.
		AllegroGraph ts = ags.renew("geospatialpolygons", AGPaths.TRIPLE_STORES);
		
		// Register namespaces
		ts.registerNamespace("ex","http://example.org/");
		
		// Get a GeoExtension instance for this store to work with geospatial features 
		GeoExtension geo = ts.getGeoExtension();
		
		// Define a geospatial subtype to encode coordinates 
		GeospatialSubtype subtype = geo.registerCartesianStriping(0,10,0,10,.1);
		geo.addSubtype(subtype);
		
		// Define a 4-sided polygon
		Polygon p1 = new Polygon(4);
		p1.setSubject(ts.addPart("!ex:p1").queryAGId());
		p1.addVertex(0,0);
		p1.addVertex(0,10);
		p1.addVertex(6,6);
		p1.addVertex(10,0);
		p1.setSubtype(subtype);
		p1.add(ts);
		
		// Define another 4-sided polygon
		Polygon p2 = new Polygon(4);
		p2.setSubject(ts.addPart("!ex:p2").queryAGId());
		p2.addVertex(0,0);
		p2.addVertex(0,10);
		p2.addVertex(4,4);
		p2.addVertex(10,0);
		p2.setSubtype(subtype);
		p2.add(ts);
		
		// Define an 8-sided polygon
		Polygon p3 = new Polygon(8);
		p3.setSubject(ts.addPart("!ex:p3").queryAGId());
		p3.addVertex(0,0);
		p3.addVertex(0,10);
		p3.addVertex(10,10);
		p3.addVertex(6,6);
		p3.addVertex(4,6);
		p3.addVertex(4,4);
		p3.addVertex(6,4);
		p3.addVertex(0,10);
		p3.setSubtype(subtype);
		p3.add(ts);
		
		// Index the store
		AGIndexAllTriples.indexAllTriplesWithTiming(ts);
		
		// Show all triples in the store
		TriplesIterator cc = ts.getStatements(null,null,null);
		AGUtils.showTriples("All triples:",cc);
		
		// Query to determine whether a point is inside the polygons
		System.out.println("Is the point (5,5) inside p1? " + p1.pointInside(5,5));
		System.out.println("Is the point (5,5) inside p2? " + p2.pointInside(5,5));
		System.out.println("Is the point (5,5) inside p3? " + p3.pointInside(5,5));
		System.out.println("Is the point (0.5,0.5) inside p3? " + p3.pointInside(0.5,0.5));
		
		// Close the store and disconnect from the server.
		ts.closeTripleStore();
		ags.disable();
	}
}
