package com.franz.agbase.examples;

import com.franz.agbase.*;

public class AGLoadRDF {

	/**
	 * Demonstrates loading a triple store from files in RDF/XML format.
	 * 
	 * @param args unused
	 * @throws AllegroGraphException
	 */
	public static void main(String[] args) throws AllegroGraphException {
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}

		// Create a fresh triple store.
		AllegroGraph ts = ags.renew("loadRDF", AGPaths.TRIPLE_STORES);

		// Load a server-side file in RDF/XML format into the store's default graph.
		loadRDFWithTiming(ts, AGPaths.dataSources("rdf-axioms.rdf"));

		// Get some triples from the default graph and show them.
		TriplesIterator it = ts.getStatements(null, null, null);
		AGUtils.showTriples(it, 20);
		
		// Load RDF data into a named graph
		URINode g = ts.createURI("http://example.org/wilburwine");
		loadRDFWithTiming(ts, AGPaths.dataSources("wilburwine.rdf"), g);

		// Get some triples from the named graph g and show them.
		it = ts.getStatements(null, null, null, g);
		AGUtils.showTriples(it, 20);
		
		// Load RDF/XML from a URL into a graph named by the URL
		ts.loadRDFXML("http://www.w3.org/TR/owl-guide/wine.rdf","source");
			
		// Get some triples from the named graph and show them.
		it = ts.getStatements(null, null, null,"<http://www.w3.org/TR/owl-guide/wine.rdf>");
		AGUtils.showTriples(it, 20);
		
		// If you have a number of files to load, it is more efficient to
		// provide an array of filenames rather than loading them separately.
		String[] files = { 
				AGPaths.dataSources("Geonames_v2.0_Lite.rdf"),
				AGPaths.dataSources("iswc-aswc-2007-complete.rdf"),
				"http://www.w3.org/TR/2004/REC-owl-guide-20040210/food.rdf"
		};
		loadRDFWithTiming(ts,files,"source");
		
		// Get some triples and note their source graph.
		it = ts.getStatements("<http://www.geonames.org/ontology#T.ISLT>", null, null, null);
		AGUtils.showTriples(it);

		// Close the triple store and disconnect from the server.
		ts.closeTripleStore();
		ags.disable();
	}
	
	/**
	 * Load a single RDF/XML file into the default graph and time the load.
	 * 
	 * @param ts A triple store 
	 * @param rdfFile A server-accessible RDF/XML file
	 * @throws AllegroGraphException
	 */
	public static void loadRDFWithTiming(AllegroGraph ts, String rdfFile) throws AllegroGraphException {
		loadRDFWithTiming(ts, rdfFile, "");
	}
	
	/**
	 * Load a single RDF/XML file into the specified graph and time the load.
	 * 
	 * @param ts A triple store 
	 * @param rdfFile A server-accessible RDF/XML file
	 * @param graph The context to load
	 * @throws AllegroGraphException
	 */
	public static void loadRDFWithTiming(AllegroGraph ts, String rdfFile, Object graph) throws AllegroGraphException {
		System.out.println("Loading RDF from " + rdfFile);
		long start = System.currentTimeMillis();
		long n = ts.loadRDFXML(rdfFile, graph);
		System.out.println("Done loading " + n + " triples in " + AGUtils.elapsedTime(start));
	}
	
	/**
	 * Load an array of RDF/XML files into the specified graph and time the load.
	 * 
	 * @param ts A triple store 
	 * @param rdfFiles An array of server-accessible RDF/XML files
	 * @param graph The context to load 
	 * @return
	 * @throws AllegroGraphException
	 */
	public static long loadRDFWithTiming(AllegroGraph ts, String[] rdfFiles, String graph) throws AllegroGraphException {
		System.out.println("Loading " + rdfFiles.length + " RDF/XML files.");
		long start = System.currentTimeMillis();
		long n = ts.loadRDFXML(rdfFiles, graph, null, false, false);
		System.out.println("Done loading " + n + " triples in " + AGUtils.elapsedTime(start));
		return n;
	}
}
