package com.franz.agbase.examples;

import com.franz.agbase.*;


public class AGPrologDisjunction {

	/**
	 * Demonstrates some basics of using disjunction and recursion in Prolog.
	 * 
	 * @param unused
	 * @throws AllegroGraphException 
	 */
	public static void main(String[] args) throws AllegroGraphException {
		
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}

		// Create a fresh triple store for this example. 
		AllegroGraph ts = ags.renew("prologdisjunction", AGPaths.TRIPLE_STORES);

		// Load the Kennedy data
		AGLoadNtriples.loadNTriplesWithTiming(ts, AGPaths.dataSources("kennedy.ntriples"));

		// Index the store for faster querying
		AGIndexAllTriples.indexAllTriplesWithTiming(ts);
		
		// Register any namespaces
		ts.registerNamespace("ex", "http://example.org/kennedy/");

		// Disjunction is represented with multiple horn clauses using
		// the <- operator to complement existing rules with the same head.  
		// Note that this group of rules also involves recursion.
		ags.addPrologRule("(<-- (ancestor ?x ?y) (q ?x !ex:has-child ?y))");
		ags.addPrologRule("(<-  (ancestor ?x ?y) (q ?x !ex:has-child ?z) (ancestor ?z ?y))");
		
		// Set up a Prolog query object
		PrologSelectQuery q = new PrologSelectQuery();
		q.setTripleStore(ts);

		// Find all ancestors of person13
		String pquery = "(ancestor ?x !ex:person13)";
		q.setVariables(new String[]{"x"});
		q.setQuery(pquery);
		ValueSetIterator it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Find all descendants of person13
		ags.addPrologRule("(<-- (descendent ?x ?y) (ancestor ?y ?x))");
		pquery = "(descendent ?x !ex:person13)";
		q.setVariables(new String[]{"x"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Close the triple store and disconnect from the server.
		ts.closeTripleStore();
		ags.disable();
	}
}
