package com.franz.agbase.examples;

import com.franz.agbase.*;

public class AGSparqlBasicGraphPatterns {

	/**
	 * Demonstrates matching of basic graph patterns in SPARQL
	 * 
	 * @param args unused
	 * @throws AllegroGraphException 
	 */
	public static void main(String[] args) throws AllegroGraphException {
		
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}
		
		// Create fresh triple-store for this example.
		AllegroGraph ts = ags.renew("sparql", AGPaths.TRIPLE_STORES);
				
		// Add data to the store
		ts.addStatement("<http://example.org/book/book1>",
				"<http://purl.org/dc/elements/1.1/title>",
				ts.createLiteral("SPARQL Tutorial"));
		
		// Define the query string: a query with a single select variable
		// and a single triple pattern in the graph pattern
		String query = 
		"SELECT ?title " + 
		"WHERE {" +
			"<http://example.org/book/book1> <http://purl.org/dc/elements/1.1/title> ?title ." +
		"}";
		
		// Query the store and show the results
		SPARQLQuery sq = new SPARQLQuery();
		sq.setTripleStore(ts);
		sq.setQuery(query);
		AGSparqlSelect.doSparqlSelect(sq);

		// Register the foaf namespace
		ts.registerNamespace("foaf", "http://xmlns.com/foaf/0.1/");
		
		// Define any blank nodes you want to reuse
		BlankNode a = (BlankNode)ts.createBNode("_:a");
		BlankNode b = (BlankNode)ts.createBNode("_:b");
		BlankNode c = (BlankNode)ts.createBNode("_:c");
		
		// Add some more data to the store
		ts.addStatement(a,"!foaf:name",ts.createLiteral("Johnny Lee Outlaw"));
		ts.addStatement(a,"!foaf:mbox","<mailto:jlow@example.com>");
		ts.addStatement(b,"!foaf:name",ts.createLiteral("Peter Goodguy"));
		ts.addStatement(b,"!foaf:mbox","<mailto:peter@example.org>");
		ts.addStatement(c,"!foaf:mbox","<mailto:carol@example.org>");

		// A query with a namespace, multiple select variables, and
		// multiple triple patterns in the graph pattern
		query =
		"PREFIX foaf: <http://xmlns.com/foaf/0.1/> " +
		"SELECT ?name ?mbox " +
		"WHERE " +
		"  { ?x foaf:name ?name . " +
		"    ?x foaf:mbox ?mbox }";
		
		// Query the store and show the results
		sq.setQuery(query);
		AGSparqlSelect.doSparqlSelect(sq);
		
		// Close the triple store and disconnect from the server.
		ts.closeTripleStore();
		ags.disable();
	}

}