package com.franz.agbase.examples;

import com.franz.agbase.*;

public class AGSparqlPlanner {

	/**
	 * Demonstrates how to set the query planner in SPARQL
	 * 
	 * @param args unused
	 * @throws AllegroGraphException 
	 */
	public static void main(String[] args) throws AllegroGraphException {
		
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}
		
		// Create fresh triple-store for this example.
		AllegroGraph ts = ags.renew("sparqlplanner", AGPaths.TRIPLE_STORES);
		
		// Load the Kennedy data set
		AGLoadNtriples.loadNTriplesWithTiming(ts, AGPaths.dataSources("kennedy.ntriples"));

		// Index the store
		AGIndexAllTriples.indexAllTriplesWithTiming(ts);
		
		// This query is written declaratively with no consideration for efficiency. 
		String query =
			"PREFIX ex: <http://example.org/kennedy/> " +
			"PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#> " +
			"SELECT ?father ?mother ?child ?univ " + 
			"WHERE {" +
				"?father rdf:type ex:person . " +
				"?mother rdf:type ex:person . " +
				"?mother ex:spouse ?father . " +
				"?child rdf:type ex:person . " +
				"?father ex:has-child ?child . " +
				"?mother ex:has-child ?child . " +
				"?father ex:alma-mater ?univ . " +
				"?child ex:alma-mater ?univ . " +
			"}";
		
		// Query the store using the (default) Variable Coverage planner; this
		// planner will do some reordering of the query's graph patterns in an
		// effort to provide results efficiently.
		SPARQLQuery sq = new SPARQLQuery();
		sq.setTripleStore(ts);
		sq.setQuery(query);
		sq.setPlanner(SPARQLQuery.PLANNER.COVERAGE);
		AGSparqlSelect.doSparqlSelect(sq);

		// Query the store using the Identity planner; this planner will not do
		// reordering of the query's graph patterns (it might take a little longer
		// because the graph patterns aren't ordered favorably).  The Identity 
		// planner is useful for applications that want the specified ordering 
		// respected.  See also AGLubm50Sparql.java for examples that use the
		// Identity planner.
		sq.setPlanner(SPARQLQuery.PLANNER.IDENTITY);
		AGSparqlSelect.doSparqlSelect(sq);

		// Close the triple store and disconnect from the server.
		ts.closeTripleStore();
		ags.disable();
	}

}