package com.franz.agbase.examples;

import com.franz.agbase.*;

public class AGSpogiCache {

	/**
	 * Demonstrates how to use the SPOGI Cache to speed up queries.
	 * 
	 * @param args unused
	 * @throws AllegroGraphException 
	 */
	public static void main(String[] args) throws AllegroGraphException {
		
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}
		
		// Create fresh triple-store for this example.
		AllegroGraph ts = ags.renew("sparqlengine", AGPaths.TRIPLE_STORES);
		
		// Show default SPOGI Cache settings
		System.out.println("Default SPOGI Cache Size: " + ts.getSPOGICacheSize());
		System.out.println("Default SPOGI Cache Enabled: " + ts.isSPOGICacheEnabled());
		
		// Enable the SPOGI Cache
		ts.setSPOGICacheSize(1000000);
		ts.setSPOGICacheEnabled(true);
		
		// Show new settings
		System.out.println("SPOGI Cache Size: " + ts.getSPOGICacheSize());
		System.out.println("SPOGI Cache Enabled: " + ts.isSPOGICacheEnabled());
		
		// Load the Kennedy data set
		AGLoadNtriples.loadNTriplesWithTiming(ts, AGPaths.dataSources("kennedy.ntriples"));

		// Index the store
		AGIndexAllTriples.indexAllTriplesWithTiming(ts);
		
		// A query written with no particular consideration for efficiency. 
		String query =
			"PREFIX ex: <http://example.org/kennedy/> " +
			"PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#> " +
			"SELECT ?father ?mother ?child ?univ " + 
			"WHERE {" +
				"?father rdf:type ex:person . " +
				"?mother rdf:type ex:person . " +
				"?mother ex:spouse ?father . " +
				"?child rdf:type ex:person . " +
				"?father ex:has-child ?child . " +
				"?mother ex:has-child ?child . " +
				"?father ex:alma-mater ?univ . " +
				"?child ex:alma-mater ?univ . " +
			"}";
		
		// Query the store (and populate the SPOGI cache)
		SPARQLQuery sq = new SPARQLQuery();
		sq.setTripleStore(ts);
		sq.setQuery(query);
		sq.setPlanner(SPARQLQuery.PLANNER.IDENTITY);
		long start = System.currentTimeMillis();
		AGSparqlSelect.doSparqlSelect(sq);
		System.out.println("done in " + AGUtils.elapsedTime(start));
		
		// Query the store again
		start = System.currentTimeMillis();
		AGSparqlSelect.doSparqlSelect(sq);
		System.out.println("done in " + AGUtils.elapsedTime(start));

		// Close the triple store and disconnect from the server.
		ts.closeTripleStore();
		ags.disable();
	}

}