package com.franz.agbase.examples;

import com.franz.agbase.*;


public class AGTemporalPointPointRelations {

	/**
	 * Demonstrates temporal queries involving relations between points.
	 * 
	 * @param unused
	 * @throws AllegroGraphException 
	 */
	public static void main(String[] args) throws AllegroGraphException {
		
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}

		// Create a fresh triple store for this example. 
		AllegroGraph ts = ags.renew("temporal", AGPaths.TRIPLE_STORES);

		// Add a datatype mapping so that XMLSchema#dateTime typed literals are 
		// loaded as date-time UPI's, enabling more efficient storage and range 
		// querying over dates.
		String[] map = new String[3];
		map[0] = "http://www.w3.org/2000/01/XMLSchema#dateTime";
		map[1] = "date-time";
		map[2] = "datatype";
		ts.addDataMapping(map);
		
		// Load some temporal data
		AGLoadNtriples.loadNTriplesWithTiming(ts, AGPaths.dataSources("temporal.nt"));

		// Index the store for faster querying
		AGIndexAllTriples.indexAllTriplesWithTiming(ts);
		
		// Register any namespaces
		ts.registerNamespace("ex", "http://example.org/");
		ts.registerNamespace("t", "http://franz.com/ns/allegrograph/3.0/temporal/");

		// Set up a Prolog query object
		PrologSelectQuery q = new PrologSelectQuery();
		q.setTripleStore(ts);
		q.setDistinct(true);

		// Query the store for points and their date-times
		String pquery = "(q ?pt !t:time ?dt)";
		q.setVariables(new String[]{"pt","dt"});
		q.setQuery(pquery);
		ValueSetIterator it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for points that are simultaneous with point pt3
		pquery = "(point-simultaneous ?pt !ex:pt3)";
		q.setVariables(new String[]{"pt"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for points that are before pt3
		pquery = "(point-before ?pt !ex:pt3)";
		q.setVariables(new String[]{"pt"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for points that have pt3 before them
		pquery = "(point-before !ex:pt3 ?pt)";
		q.setVariables(new String[]{"pt"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for points that are after pt6
		pquery = "(point-after ?pt !ex:pt6)";
		q.setVariables(new String[]{"pt"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for points that have pt6 after them
		pquery = "(point-after !ex:pt6 ?pt)";
		q.setVariables(new String[]{"pt"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Close the triple store and disconnect from the server.
		ts.closeTripleStore();
		ags.disable();
	}

}
