/**
 * 
 */
package com.franz.agbase;

import com.franz.agbase.AllegroGraphException;
import com.franz.ag.UPI;
import com.franz.agbase.impl.GeospatialSubtypeImpl;
import com.franz.agbase.transport.AGConnector;
import com.franz.agbase.util.AGC;
import com.franz.agbase.util.AGInternals;

/**
 * This class is used to represent a collection of point coordinates that form
 * a polygon in 2-dimensional space.
 * <p>
 * A Polygon is represented in a triple store as a collection of triples, all with
 * the same subject.  The predicates are generated by the implementation and 
 * identify the point coordinates.  The objects of the triples are encoded 
 * vertex coordinates.
 * 
 * @author mm
 *
 */
public class Polygon {
	
	private AGInternals pag = null;
	private double[] points;   // { x0, y0, x1, y1, ... }
	private int i = 0;
	private UPI subject = null;
	private boolean modified = true;
	private GeospatialSubtypeImpl subtype;
	
	/**
	 * Get the GeospatialSubtype associated with all the points in the Polygon instance.
	 * @return the GeospatialSubtype
	 */
	public GeospatialSubtype getSubtype() {
		return subtype;
	}

	/**
	 * Set the GeospatialSubtype associated with all the points in the Polygon instance.
	 * @param subtype2 the GeospatialSubtype
	 */
	public void setSubtype(GeospatialSubtype subtype2) {
		modified = true;
		this.subtype = (GeospatialSubtypeImpl) subtype2;
	}

	/**
	 * Get the array of coordinates for the points in the Polygon.
	 * @return an array of alternating x and y coordinates
	 */
	public double[] getPoints() {
		return points.clone();
	}

	/**
	 * Query the state of a Polygon instance.
	 * @return true if the Polygon contains changes not transmitted to the server.
	 */
	public boolean getModified () { return modified; }
	
	/**
	 * Query the subject that identifies the triples in a polygon.
	 * @return the subject
	 */
	public UPI getSubject() {
		return subject;
	}

	/**
	 * Set the subject that identifies the triples in a polygon.
	 * If the subject is null, a new blank node will be assigned to the Polugon
	 * when it is added to a triple store.
	 * @param subject the subject to set
	 */
	public void setSubject(UPI subject) {
		modified = true;
		this.subject = subject;
	}
	
	/**
	 * Create an empty Polygon instance with room for a given number of vertices.
	 * @param vertices the number of vertices in the Polygon.
	 */
	public Polygon ( int vertices ) {
		modified = true;
		points = new double[2*vertices];
	}

	
	/**
	 * Add one pair of vertex coordinates to the Polygon.
	 * @param x
	 * @param y
	 */
	public void addVertex ( double x, double y ) {
		modified = true;
		points[i] = x;
		i++;
		points[i] = y;
		i++;
	}	
	
	/**
	 * Add the polygon to a triple store.
	 * @param ag The triple store.
	 * @return the UPI that identifies the triples in the polygon.
	 * @throws AllegroGraphException
	 */
	public UPI add ( AllegroGraph ag ) throws AllegroGraphException {
		if ( pag!=null ) throw new IllegalStateException("Already added to store " + pag);
		if ( i<points.length ) throw new IllegalStateException("Incomplete Polygon");
		if ( null==subtype ) 
			throw new IllegalStateException("Cannot add Polygon without a subtype");
		pag = ag;
		Object[] v = ag.verifyEnabled().applyAGFn(pag, AGC.AG_ADD_POLYGON, 
				new Object[] { subject, new Long(subtype.getLocal(ag.ags)), points });
		subject = (UPI) v[0];
		modified = false;
		return subject;
	}
	
	/**
	 * Retrieve the coordinates of this polygon from the triple store.
	 * @param ag the triple store
	 * @throws AllegroGraphException
	 */
	public void get ( AllegroGraph ag ) throws AllegroGraphException {
		if ( null==subject )
			throw new IllegalStateException("Cannot locate Polygon with null subject");
		pag = ag;
		Object[] v = ag.verifyEnabled().applyAGFn(pag, AGC.AG_GET_POLYGON, 
				new Object[] { subject });
		//long subix = AGConnector.longValue(v[0]);
		points = (double[]) v[0];
		i = points.length;
		modified = false;
	}
	
	
	/**
	 * Query whether a point is inside a polygon.
	 * @param x
	 * @param y
	 * @return true if the point is inside
	 * @throws AllegroGraphException
	 */
	public boolean pointInside ( double x, double y ) throws AllegroGraphException {
		Object[] v = pag.verifyEnabled().applyAGFn(pag, AGC.AG_POINTS_INSIDE_POLY, 
				new Object[] { points, new Double(x), new Double(y) });
		return 1==AGConnector.longValue(v[0]);
	}
	
	/**
	 * Query whether some points are inside a polygon.
	 * @param xy An array of alternating x and y coordinates
	 * @return an array of boolean values. 
	 *    The length of the array is half the length of the xy array. 
	 *    Each boolean value is true or false if the corresponding point is inside
	 *    or outside the Polygon.
	 * @throws AllegroGraphException
	 */
	public boolean[] pointsInside (  double[] xy ) throws AllegroGraphException {
		Object[] v = pag.verifyEnabled().applyAGFn(pag, AGC.AG_POINTS_INSIDE_POLY, 
				new Object[] { points, xy  });
		long[] w = AGConnector.longArray(v[0]);
		boolean[] r = new boolean[w.length];
		for (int i = 0; i < r.length; i++) {
			r[i] = 1==w[i];
		}
		return r;
	}
	
	/**
	 * Query whether some points are inside a polygon.
	 * @param x An array of x coordinates
	 * @param y An array of y coordinates
	 *     The two arrays must of the same length.
	 * @return an array of boolean values. 
	 *    The length of the array is equal to the length of the argument array.
	 *    Each boolean value is true or false if the corresponding point is inside
	 *    or outside the Polygon.
	 * @throws AllegroGraphException
	 */
	public boolean[] pointsInside ( double[] x, double[] y ) throws AllegroGraphException {
		Object[] v = pag.verifyEnabled().applyAGFn(pag, AGC.AG_POINTS_INSIDE_POLY, 
				new Object[] { points, x, y  });
		long[] w = AGConnector.longArray(v[0]);
		boolean[] r = new boolean[w.length];
		for (int i = 0; i < r.length; i++) {
			r[i] = 1==w[i];
		}
		return r;
	}
	
}