
// ***** BEGIN LICENSE BLOCK *****
// Version: MPL 1.1
//
// The contents of this file are subject to the Mozilla Public License Version
// 1.1 (the "License"); you may not use this file except in compliance with
// the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
// for the specific language governing rights and limitations under the
// License.
//
// The Original Code is the AllegroGraph Java Client interface.
//
// The Original Code was written by Franz Inc.
// Copyright (C) 2006 Franz Inc.  All Rights Reserved.
//
// ***** END LICENSE BLOCK *****

package com.franz.ag.impl;

import org.openrdf.model.GraphException;
import org.openrdf.model.URI;
import org.openrdf.sesame.sail.StatementIterator;

import com.franz.ag.AllegroGraph;
import com.franz.ag.AllegroGraphException;
import com.franz.ag.Node;
import com.franz.ag.UPI;
import com.franz.agbase.URINode;
import com.franz.agbase.impl.ValueObjectImpl;

/**
 * This class represents an instance of a labelled resource node in AllegroGraph.
 * <p>
 * The AllegroGraph object defines two slots, id and uri.  Both slots are copied 
 * to the Java instance.
 * <p>
 * The URI member may be a lazy value in the Java instance.  If queryURI returns
 * null, getURI() will need a round-trip to the triple store to fetch the
 * actual value.
 * <p>
 * There is no public constructor.  Node instances are created by calls
 * to AllegroGraph methods.
 */
public class NodeImpl 
    extends ResourceNodeImpl
    implements Node
{

    /**
	 * 
	 */
	private static final long serialVersionUID = 8675791173081086903L;

	com.franz.agbase.URINode baseNode;

//	 Use package access here because only use should be in AGFactory
	NodeImpl ( AllegroGraph ts, UPI i, String u ) {
	super();
	owner = ts; 
	baseInstance = (ValueObjectImpl) com.franz.agbase.impl.AGFactory.makeNode(ts.agbase, i, u);
	baseNode = (URINode) baseInstance;
	}
	
	NodeImpl ( AllegroGraph ts, com.franz.agbase.URINode base ) {
		owner = ts; 
		baseInstance = (ValueObjectImpl) base;
		baseNode = (URINode) baseInstance;
	}
	
	
	/**
	 * Retrieve the AllegroGraph ID number of the Node.
	 * @return the ID number
	 * <p>
	 * If the Node is already registered in the AG triple store, return the locally
	 * cached value of the ID number.  Otherwise, register the Node in the
	 * AG triple store and return the new ID number.
	 * @throws AllegroGraphException
	 */
	public UPI getAGId () throws AllegroGraphException {
		return baseNode.getAGId();
	}
	

    /**
     * Retrieve the URI string associated with the node instance.
     * @return A string or null.
     * If the returned value is null, the actual value must be obtained
     * by calling getURI().
     */
    public String queryURI () { return baseNode.queryURI(); }

  

    /**
     * Retrieve the local name component of the URI string associated 
     * with the node instance.
     * @return A string.
     * If the value is not in the Java cache, retrieve it from the triple store.
     */
    public String getLocalName ()
    { return baseNode.getLocalName();
    }

    /**
     * Retrieve the namespace component of the URI string associated 
     * with the node instance.
     * @return A string or null if the URI does not have a namespace component.
     * If the value is not in the Java cache, retrieve it from the triple store.
     */
    public String getNamespace ()
    { return baseNode.getNamespace();
    }

    /**
     * Retrieve the URI string associated with the node instance.
     * @return A string.
     * If the value is not in the Java cache, retrieve it from the triple store.
     * <p>
     * Defined in interface org.openrdf.model.URI
     */
    public String toString() {
    	return baseNode.toString();
    } 
    
    // These are not referenced, nad should not be because the work
    // needs to be more complex if done at all.
//    String stringRef() {
//    	if ( uri==null ) throw new IllegalStateException("Cannot stringify this node");
//    	return "<" + uri + ">";
//    }
//    
//    static String stringRef ( String uri ) {
//    	if ( uri==null ) throw new IllegalArgumentException("Cannot reference a null URI");
//    	return "<" + uri + ">";
//    }
    
    /**
     * Implement equality for Node instances.
     * <p>
     * Two Node instances are equal if both are registered in the
     * AllegroGraph triple store and  they have identical
     * AllegroGraph part id numbers.
     * <p>
     * Otherwise, the string representations are compared.
     */
    public boolean equals(Object other) {
    	if ( other==null ) return false;
    	if ( this==other ) return true;
    	switch(sameAGId(other)) {
    	case 1: return true;
    	case 0: return false;
    	}
        if (other instanceof URI) {
           // as per the OpenRDF model API:
           return toString().equals(other.toString());
        }
		return false;
    }

    /**
     * Compute the hashcode of a Node instance.
     * <p>
     * The hashcode of a Node instance is the hashcode
     * of its string representation.
     */
    public int hashCode() {
        // as per the OpenRDF model API:
        return toString().hashCode();
    }

	public StatementIterator getPredicateStatements() throws GraphException {
		try {
			return owner.getStatements(null, this, null);
		} catch (AllegroGraphException e) {
			throw new GraphException(e);
		}
	}

	public String getURI() { return toString(); }
	
	/**
     * Add this node to the AllegroGraph triple store.
     * If the node already is in the triple store, do nothing.
     * <p>
     * A Node instance is in the triple store if queryAGId() returns
     * a non-null value.
	 * @throws AllegroGraphException 
     *
     */
    public void add () throws AllegroGraphException {
    	getAGId();
    }
    

}
