package com.franz.agbase.examples;



import java.io.File;
import java.io.FilenameFilter;

import com.franz.agbase.*;

public class AGLubm8000Build {

	public static int port = 4126;
	public static String store = "/stores/LUBM-8000";
	public static String datadir = "/lubm8000.ntriples/data/"; 
	public static String ontologyfile = "/lubm8000.ntriples/ontology/lubm-franz.nt"; 
	public static String fileext = ".ntriples";
	public static long expectedResources = 400000000;
	public static long chunkSize = 10000000;
	
	/**
	 * Demonstrates loading and indexing a triple store with the LUBM 8000 
	 * benchmark data.
	 * 
	 * The data that is necessary to run this example can be generated using
	 * the LUBM UBA Generator (see http://swat.cse.lehigh.edu/projects/lubm/)
	 * 
	 * with arguments:
	 * 
	 * -univ 8000 -index 0 -seed 0 -onto "http://lubm.franz.com"
	 * 
	 * Please refer to the file AGLubm8000Prolog.java to run these queries
	 * using RDFS++ reasoning.
	 * 
	 * @throws AllegroGraphException 
	 */		
	public static void main(String[] args) throws AllegroGraphException {

		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.setPort(port);
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}
		System.out.println("Setting default expected resources: " + expectedResources);
		ags.setDefaultExpectedResources(expectedResources);
		System.out.println("Setting chunk size: " + chunkSize);
		ags.setChunkSize(chunkSize);
		
		// Create fresh triple-store
		System.out.println("Creating store: " + store);
		AllegroGraph ts = ags.renew(store, null);

		// Find files having the given file extension in the datadir.
		System.out.println("Finding " + fileext + " files in " + datadir);
	    File dir = new File(datadir);
	    if (!dir.isDirectory()) {
	    	System.out.println("datadir does not exist: " + datadir);
	    	System.exit(1);
	    }
	    FilenameFilter filter = new FilenameFilter() {
	        public boolean accept(File dir, String name) {
	            return name.endsWith(fileext);
	        }
	    };
	    File[] files = dir.listFiles(filter);
	    String[] filenames = new String[files.length];
	    for (int i=0; i<files.length; i++) {
	    	filenames[i] = files[i].getPath();
	    }
	    
	    // Load the files
		System.out.println("Loading " + filenames.length + " " + fileext + " files from " + datadir);
		long begin = System.nanoTime();
		long n = ts.loadNTriples(filenames);
		long delta = (System.nanoTime() - begin);
		System.out.println("Done loading " + n + " triples in " + (delta/1000000000.0) + " seconds.");
	    
	    // Assert that GraduateStudent is a subClassOf Student
	    // This helps reduce LUBM to an RDFS++ problem
		ts.addStatement("<http://lubm.franz.com#GraduateStudent>",
	    		"<http://www.w3.org/2000/01/rdf-schema#subClassOf>",
	    		"<http://lubm.franz.com#Student>");
	    
		// Index the store
		ts.setIndexFlavors(new String[]{"spogi","posgi","ospgi"});
		begin = System.nanoTime();
		ts.indexAllTriples();
		delta = (System.nanoTime() - begin);
		System.out.println("Done indexing in " + (delta/1000000000.0) + " seconds.");

		// Close the store and disconnect from the server
		ts.closeTripleStore();
		ags.disable();
 	}
	
}
