package com.franz.agbase.examples;

import com.franz.agbase.*;

public class AGPrologFunctorQ {

	/**
	 * Demonstrates basics of using the functor q in PrologSelect queries.
	 * 
	 * @param args unused
	 * @throws AllegroGraphException 
	 */
	public static void main(String[] args) throws AllegroGraphException {
		
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}

		// Create a fresh triple store for this example. 
		AllegroGraph ts = ags.renew("functorq", AGPaths.TRIPLE_STORES);

		// Load the Kennedy data
		AGLoadNtriples.loadNTriplesWithTiming(ts, AGPaths.dataSources("kennedy.ntriples"));

		// Index the store for faster querying
		AGIndexAllTriples.indexAllTriplesWithTiming(ts);
		
		// Register any namespaces
		ts.registerNamespace("ex", "http://example.org/kennedy/");

		// Set up a Prolog query object
		PrologSelectQuery q = new PrologSelectQuery();
		q.setTripleStore(ts);

		// Find all of the children of person1
		String pquery = "(q !ex:person1 !ex:has-child ?x)";
		q.setVariables(new String[]{"x"});
		q.setQuery(pquery);
		ValueSetIterator it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);

		// Find all of the grand children of person1
		pquery = "(q !ex:person1 !ex:has-child ?x)(q ?x !ex:has-child ?y)";
		q.setVariables(new String[]{"y"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);

		// Find all of the grand children of person1 that have spouses
		pquery = "(q !ex:person1 !ex:has-child ?x)"
				+ "(q ?x !ex:has-child ?y)" + "(q ?y !ex:spouse ?z)";
		q.setVariables(new String[]{"y","z"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);

		// Find parents and children with the same first name 
		pquery = "(q ?x !ex:first-name ?n1)" +
			     "(q ?x !ex:has-child ?y)" +
			     "(q ?y !ex:first-name ?n2)" +
			     "(= ?n1 ?n2)";
		q.setVariables(new String[]{"x","y"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);

		// Find parents and children that went to ivy league schools
		pquery = "(q ?x !ex:alma-mater ?am)" +
			     "(q ?am !ex:ivy-league !ex:true)" +
			     "(q ?x !ex:has-child ?y)" +
			     "(q ?y !ex:alma-mater ?am2)" +
			     "(q ?am2 !ex:ivy-league !ex:true)";
		q.setVariables(new String[]{"x","y"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Find those that attended the same ivy league school
		pquery = "(q ?x !ex:alma-mater ?am)" +
			     "(q ?am !ex:ivy-league !ex:true)" +
			     "(q ?x !ex:has-child ?y)" +
			     "(q ?y !ex:alma-mater ?am)";
		q.setVariables(new String[]{"x","y"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Close the triple store and disconnect from the server.
		ts.closeTripleStore();
		ags.disable();
	}
	
}