package com.franz.agbase.examples;

import com.franz.agbase.*;

public class AGPrologQueryPlanner {

	/**
	 * Demonstrates some basics of using the Prolog query planner.
	 * 
	 * @param args unused
	 * @throws AllegroGraphException 
	 */
	public static void main(String[] args) throws AllegroGraphException {
		
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}

		// Create a fresh triple store for this example. 
		AllegroGraph ts = ags.renew("queryplanner", AGPaths.TRIPLE_STORES);

		// Load the Kennedy data
		AGLoadNtriples.loadNTriplesWithTiming(ts, AGPaths.dataSources("kennedy.ntriples"));

		// Index the store for faster querying
		AGIndexAllTriples.indexAllTriplesWithTiming(ts);
		
		// Register any namespaces
		ts.registerNamespace("ex", "http://example.org/kennedy/");

		// Set up a Prolog query object
		PrologSelectQuery q = new PrologSelectQuery();
		q.setTripleStore(ts);

		// Find parents and children that went to ivy league schools
		q.setVariables(new String[]{"x","y"});
		String pquery = "(q ?x !ex:alma-mater ?am)" +
			     "(q ?am !ex:ivy-league !ex:true)" +
			     "(q ?x !ex:has-child ?y)" +
			     "(q ?y !ex:alma-mater ?am2)" +
			     "(q ?am2 !ex:ivy-league !ex:true)";
		q.setQuery(pquery);
		
		// Prepare a plan and show it (but don't run it yet)
		System.out.println("Prepare a query plan: " + q.preparePlan());
		System.out.println(q.showPlan());
		
		// Now run the plan
		ValueSetIterator it = q.runPlan();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Show some plan options
		System.out.println(q.isRemoveRTF());
		System.out.println(q.isReorder());
		System.out.println(q.isSavedPlan());
		System.out.println(q.isUseMaps());
		System.out.println(q.isUseTransforms());
		
		// Modify some plan options, run and save the new plan
		q.setRemoveRTF(false);
		q.setReorder(false);
		q.setUseMaps(false);
		q.setUseTransforms(false);
		if (q.runAndSavePlan()) {
			System.out.println(q.showPlan());
			it = q.getResult();
			System.out.println("Results for " + pquery);
			AGUtils.showResults(it);
		}
		
		// Close the triple store and disconnect from the server.
		ts.closeTripleStore();
		ags.disable();
	}
	
}