package com.franz.agbase.examples;

import com.franz.agbase.*;

public class AGFederationBasics {

	/**
	 * Demonstrates some basics of federating triple stores.
	 * 
	 * @param args unused
	 * @throws AllegroGraphException
	 */
	public static void main(String[] args) throws AllegroGraphException {
		
		// Connect to the default server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem.", e);
		}

		// Create triple stores for this example
		AllegroGraph ts1 = ags.renew("store1", AGPaths.TRIPLE_STORES);
		AllegroGraph ts2 = ags.renew("store2", AGPaths.TRIPLE_STORES);
		
		// Create a federated store consisting of the first 2 open stores above 
		AllegroGraph[] parts = {ts1,ts2};
		AllegroGraph fed = ags.federate("federation", parts, true);
		
		// Show stores in the federation 
		AllegroGraph[] stores = fed.getStores();
		AGUtils.printObjectArray("federation stores:", stores);

		// Register namespaces with each leaf store
		ts1.registerNamespace("ex", "http://example.org/");
		ts2.registerNamespace("ex", "http://example.org/");
		
		// Add some triples to the default graph of each store
		ts1.addStatement("!ex:A", "!rdfs:subClassOf", "!ex:B");
		ts2.addStatement("!ex:B", "!rdfs:subClassOf", "!ex:C");
		
		// You cannot add triples to a federation this way,
		// you must add directly to one of its leaf stores
		//fed.addStatement("!ex:a", "!rdf:type", "!ex:A");
		
		// Show all triples in the default graph of store 1.
		TriplesIterator cc = ts1.getStatements(null, null, null);
		System.out.println("Showing all triples in the default graph of store1:");
		AGUtils.showTriples(cc);
		
		// Show all triples in the default graph of store 2.
		cc = ts2.getStatements(null, null, null);
		System.out.println("Showing all triples in the default graph of store2:");
		AGUtils.showTriples(cc);
		
		// Show all triples in the default graph of the federation.
		// Note that the federation's default graph is empty, and
		// is distinct from the default graph of each leaf store
		cc = fed.getStatements(null, null, null);
		System.out.println("Retrieving all triples in the default graph of the federated store (should be empty):");
		AGUtils.showTriples(cc);
		
		// Retrieve all triples in the federation.
		// Note that we're using a wildcard on the graph argument.
		cc = fed.getStatements(null, null, null, null);
		System.out.println("Retrieving all triples in the federated store:");
		AGUtils.showTriples(cc);
		
		// Show all triples inferred from the federation.
		cc = fed.getStatements(true, null, null, null, null);
		System.out.println("Showing all triples inferred from the federated store:");
		AGUtils.showTriples(cc);
		
		// Close all stores and disconnect from the server
		fed.closeTripleStore();
		ts1.closeTripleStore();
		ts2.closeTripleStore();
		ags.disable();
	}
	
}
