package com.franz.agbase.examples;

import com.franz.agbase.*;

public class AGIndexAutomatically {

	/**
	 * Demonstrates some basics of indexing triples automatically.
	 * 
	 * @param args unused
	 * @throws AllegroGraphException
	 */
	public static void main(String[] args) throws AllegroGraphException {
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}

		// Set the index chunk size (in triples) -- memory is allocated in 
		// chunks as the number of triples in a store grows
		long chunkSize = ags.getChunkSize();
		System.out.println("Initial index chunk size: " + chunkSize);
		ags.setChunkSize(1000);
		System.out.println("Setting index chunk size: " + ags.getChunkSize());
		
		// Create a fresh triple store
		AllegroGraph ts = ags.renew("wilburwine", AGPaths.TRIPLE_STORES);

		// Set the unindexed triples threshold -- if the number of unindexed 
		// triples reaches this threshold, the server indexes.  Note that the
		// threshold defaults to 0, which means no automatic indexing.
		int unindexedThreshold = ts.getUnindexedThreshold();
		System.out.println("Current unindexed triples threshold: " + unindexedThreshold);
		ts.setUnindexedThreshold(500);
		System.out.println("Setting unindexed triples threshold: " + ts.getUnindexedThreshold());
		
		// Set the unmerged threshold -- when the number of unmerged index 
		// chunks reaches this threshold, the server merges them.  Note that 
		// the threshold defaults to 0, which means no automatic merging.
		int unmergedThreshold = ts.getUnmergedThreshold();
		System.out.println("Current unmerged threshold: " + unmergedThreshold);
		ts.setUnmergedThreshold(2);
		System.out.println("Setting unmerged threshold: " + ts.getUnmergedThreshold());
		
		// Load the wilburwine dataset
		String ntripleFile = AGPaths.dataSources("wilburwine.ntriples");		
		System.out.println("Loading N-Triples " + ntripleFile);
		AGLoadNtriples.loadNTriplesWithTiming(ts,ntripleFile);
		
		// Check to see if indexing occurred automatically.  The server polls 
		// every 2 seconds by default to determine if indexing or merging is
		// required.
		System.out.println("Checking to see if indexing occurred automatically.");
		System.out.println("Unmerged Chunks: " + ts.getUnmergedCount());
		System.out.println("Unindexed triples: " + ts.getUnindexedTripleCount());
		
		// Sleep for 3 seconds until the server polls and indexes
		System.out.println("Waiting 5 seconds ...");
		try {
			Thread.sleep(5000);
		} catch (Exception e) {
			System.out.println(e.getMessage());
		}
		
		// Now check again
		System.out.println("Checking again to see if indexing occurred automatically.");
		System.out.println("Unmerged Chunks: " + ts.getUnmergedCount());
		System.out.println("Unindexed triples: " + ts.getUnindexedTripleCount());
		
		// Return the server to its prior settings
		System.out.println("Returning the server to its prior settings.");
		ags.setChunkSize(chunkSize);
		ts.setUnindexedThreshold(unindexedThreshold);
		ts.setUnmergedThreshold(unmergedThreshold);
		
		// Close the triple store and disconnect from the server.
		ts.closeTripleStore();
		ags.disable();
	}

}
