package com.franz.agbase.examples;


import java.io.File;
import java.io.FilenameFilter;

import com.franz.agbase.*;

public class AGLubmBuild {

	// args that must be provided
	public static int port = 0;
	public static String datadir = null; 
	public static String store = null;
	
	// optional args with defaults
	public static String ontologyurl = "http://www.lehigh.edu/~zhp2/2004/0401/univ-bench.owl"; 
	public static String fileext = ".owl";
	public static String ubnamespace = "http://www.lehigh.edu/~zhp2/2004/0401/univ-bench.owl#";
	public static boolean useRapper = false;
	public static long expectedResources = 3000000;
	public static long chunkSize = 8000000;
	
	/**
	 * Demonstrates loading and indexing a triple store with LUBM benchmark data.
	 * 
	 * The data that is necessary to run this example can be generated using
	 * the LUBM UBA Generator (see http://swat.cse.lehigh.edu/projects/lubm/)
	 * 
	 * with arguments:
	 * 
	 * -univ <size> -index 0 -seed 0
	 * 
	 * For faster loading, you may want to use Rapper to convert the generated
	 * OWL files into NTriples files and load the latter.
	 * 
	 * Please refer to the file AGLubmProlog.java to run the queries against
	 * the store.
	 * 
	 * @throws AllegroGraphException 
	 */	
	public static void main(String[] args) throws AllegroGraphException {

		processArgs(args);
		
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.setPort(port);
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}
		ags.setDefaultExpectedResources(expectedResources);
		ags.setChunkSize(chunkSize);
		
		// Create fresh triple-store
		System.out.println("Creating store: " + store);
		AllegroGraph ts = ags.renew(store, null);

		// Find files having the given file extension in the datadir.
	    File dir = new File(datadir);
	    if (!dir.isDirectory()) {
	    	System.out.println("datadir does not exist: " + datadir);
	    	System.exit(1);
	    }
	    FilenameFilter filter = new FilenameFilter() {
	        public boolean accept(File dir, String name) {
	            return name.endsWith(fileext);
	        }
	    };
	    File[] files = dir.listFiles(filter);
	    String[] filenames = new String[files.length];
	    for (int i=0; i<files.length; i++) {
	    	filenames[i] = files[i].getPath();
	    }
	    
	    // Fetch and load the LUBM ontology
	    System.out.println("Fetching the LUBM ontology " + ontologyurl);
	    long begin = System.nanoTime();
	    ts.loadRDFXML(ontologyurl, null, null, false, false);
	    long delta = (System.nanoTime() - begin);
	    System.out.println("Done fetching and loading " + ts.numberOfTriples() + " triples in " + (delta/1000000000.0) + " seconds.");
	    
	    // Load the data files
		System.out.println("Loading " + filenames.length + " files (" + fileext + ") from " + datadir);
		begin = System.nanoTime();
		long n = 0;
		if (fileext.startsWith(".nt")) {
			n = ts.loadNTriples(filenames);
		} else if (fileext.equals(".owl") || fileext.equals(".rdf")) {
			n = ts.loadRDFXML(filenames, null, null, useRapper, false);
		} else {
			throw new AllegroGraphException("Don't know how to load " + fileext + " files.");
		}
		delta = (System.nanoTime() - begin);
		System.out.println("Done loading " + n + " triples in " + (delta/1000000000.0) + " seconds.");
	    
	    // Assert that GraduateStudent is a subClassOf Student.
	    // This helps reduce LUBM to an RDFS++ problem
		ts.addStatement("<" + ubnamespace + "GraduateStudent>",
	    		"<http://www.w3.org/2000/01/rdf-schema#subClassOf>",
	    		"<" + ubnamespace + "Student>");
	    
		// Index the store
		ts.setIndexFlavors(new String[]{"spogi","posgi","ospgi"});
		begin = System.nanoTime();
		ts.indexAllTriples();
		delta = (System.nanoTime() - begin);
		System.out.println("Done indexing in " + (delta/1000000000.0) + " seconds.");

		// Close the store and disconnect from the server
		ts.closeTripleStore();
		ags.disable();
 	}
	
	static void showUsage() {
		System.out.println("Usage: ");
		System.out.println("  required: -port <int> [e.g. -port 4567]");
		System.out.println("  required: -datadir <directory> [e.g. -datadir /data/lubm-50]");
		System.out.println("  required: -store <directory> [e.g. -store /stores/LUBM-50]");
		System.out.println("  optional: -fileext <ext> [e.g. -fileext " + fileext + "]");
		System.out.println("  optional: -ontologyurl <url> [e.g. -ontologyurl \"" + ontologyurl + "\"]");
		System.out.println("  optional: -ub <namespace> [defaults to \"" + ubnamespace + "\"]");
		System.out.println("  optional: -expectedResources <long> [defaults to " + expectedResources + "]");
		System.out.println("  optional: -chunkSize <long> [defaults to " + chunkSize + "]");
		System.out.println("  optional: -useRapper <boolean> [defaults to " + useRapper + "]");
		System.out.println("  optional: -help or -? prints this usage message");
		System.exit(1);
	}
	
	/**
	 * Process command line args.
	 * 
	 * @param args
	 */
	public static void processArgs(String[] args) {
		for (int i=0; i<args.length; i++) {
			String arg = args[i];
			if (arg.equals("-store")) {
				store = args[++i];
			} else if (arg.startsWith("-ub")) {
				ubnamespace = args[++i];
			} else if (arg.startsWith("-data")) {
				datadir = args[++i];
			} else if (arg.startsWith("-onto")) {
				ontologyurl = args[++i];
			} else if (arg.startsWith("-file")) {
				fileext = args[++i];
			} else if (arg.equalsIgnoreCase("-useRapper")) {
				useRapper = Boolean.parseBoolean(args[++i]);
			} else if (arg.equals("-port")) {
				try {
					port = Integer.parseInt(args[++i]);
				} catch (NumberFormatException e) {
					System.out.println("Invalid port: " + arg);
					showUsage();
				}
			} else if (arg.startsWith("-expected")) {
				try {
					expectedResources = Long.parseLong(args[++i]);
				} catch (NumberFormatException e) {
					System.out.println("Invalid expectedResources: " + arg);
					showUsage();
				}
			} else if (arg.startsWith("-chunk")) {
				try {
					expectedResources = Long.parseLong(args[++i]);
				} catch (NumberFormatException e) {
					System.out.println("Invalid expectedResources: " + arg);
					showUsage();
				}
			} else {
				if (!arg.endsWith("help")&&!arg.endsWith("?")) {
					System.out.println("Unrecognized arg: " + arg);
				}
				showUsage();
			}
		}
		if (port==0 || store==null || datadir==null) {
			showUsage();
		}
	}	
		
}
