package com.franz.agjena.examples;


import com.franz.agbase.AllegroGraph;
import com.franz.agbase.AllegroGraphConnection;
import com.franz.agbase.AllegroGraphException;
import com.franz.agbase.examples.AGPaths;
import com.franz.agjena.AllegroGraphGraphMaker;
import com.franz.agjena.AllegroGraphModel;
import com.hp.hpl.jena.graph.Graph;
import com.hp.hpl.jena.graph.GraphMaker;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;
import com.hp.hpl.jena.vocabulary.VCARD;

public class AGJenaStatements {

	/**
	 * Demonstrates Jena Tutorial 03 using AllegroGraph -- working with Statements.
	 *
	 * @param args unused
	 * @throws AllegroGraphException
	 */
	public static void main(String[] args) throws AllegroGraphException {
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}

		// Create a fresh triple store
		AllegroGraph ts = ags.renew("jenatutorial03", AGPaths.TRIPLE_STORES);

		// Create a Graph maker for the store
	    GraphMaker maker = new AllegroGraphGraphMaker(ts);
    		
        // some definitions
        String personURI    = "http://somewhere/JohnSmith";
        String givenName    = "John";
        String familyName   = "Smith";
        String fullName     = givenName + " " + familyName;
        
        // create an empty graph and a model for it:
        Graph graph = maker.getGraph();
        Model model = new AllegroGraphModel(graph);
        
        // create the resource
        //   and add the properties cascading style
        Resource johnSmith = model.createResource(personURI)
         		.addProperty(VCARD.FN, fullName)
         			.addProperty(VCARD.N, 
                              model.createResource()
                                   .addProperty(VCARD.Given, givenName)
                                   .addProperty(VCARD.Family, familyName));
        
        johnSmith.addLiteral(VCARD.FN, "John J. Smith");
        
        // list the statements in the graph
        StmtIterator iter = model.listStatements();
        
        // print out the predicate, subject and object of each statement
        while (iter.hasNext()) {
            Statement stmt      = iter.nextStatement();
            Resource  subject   = stmt.getSubject();
            Property  predicate = stmt.getPredicate();
            RDFNode   object    = stmt.getObject();
            System.out.print(subject.toString());
            System.out.print(" " + predicate.toString() + " ");
            if (object instanceof Resource) {
                System.out.print(object.toString());
            } else {
                // object is a literal
                System.out.print(" \"" + object.toString() + "\"");
            }
            System.out.println(" .");
        }
        
        String query = "select ?s ?p ?o where {?s ?p ?o }";
		AGJenaUtils.doQuery(query, model);
 
		// Close the triple store and disconnect from the server
		ts.closeTripleStore();
		ags.disable();
	}

}
