
// ***** BEGIN LICENSE BLOCK *****
// Version: MPL 1.1
//
// The contents of this file are subject to the Mozilla Public License Version
// 1.1 (the "License"); you may not use this file except in compliance with
// the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
// for the specific language governing rights and limitations under the
// License.
//
// The Original Code is the AllegroGraph Java Client interface.
//
// The Original Code was written by Franz Inc.
// Copyright (C) 2006 Franz Inc.  All Rights Reserved.
//
// ***** END LICENSE BLOCK *****

package com.franz.ag.impl;

import org.openrdf.model.URI;

import com.franz.ag.AllegroGraph;
import com.franz.ag.AllegroGraphException;
import com.franz.ag.Literal;
import com.franz.ag.UPI;
import com.franz.agbase.URINode;
import com.franz.agbase.impl.LiteralNodeImpl;
import com.franz.agbase.impl.ValueObjectImpl;
import com.franz.agbase.impl.LiteralNodeImpl.Lang;


/**
 * This class represents an instance of a literal node in AllegroGraph.
 * <p>
 * The AllegroGraph object defines two slots, id and label.  Both slots are copied 
 * to the Java instance.
 * <p>
 * The label member may be a lazy value in the Java instance.  If queryLabel returns
 * null, getLabel() must make a round-trip to the triple store for the actual value.
 * <p>
 * There is no public constructor.  Literal instances are created by calls
 * to AllegroGraph methods.
 */
public class LiteralImpl
    extends ValueNodeImpl
    implements org.openrdf.model.Literal, Literal
{

    /**
	 * 
	 */
	private static final long serialVersionUID = -3106405889572308071L;
    
    /**
     *    typeId     type
     *      null     null    Not a typed literal
     *      null    string   Typed literal, string is URI
     *    res-UPI    null    Typed literal, UPI is type
     *    res-UPI   string   Typed literal, UPI is type, string is URI
     */
    public UPI typeId;
   
    
    public static final Lang LANG_KNOWN = Lang.KNOWN;
    public static final Lang LANG_NONE  = Lang.NONE;

    public LiteralNodeImpl baseLiteral;
    
//  Use package access here because only use should be in AGFactory
    LiteralImpl ( AllegroGraph ts, UPI i, String newLabel,
    				UPI newTypeId, String newType, 
    				Lang newLangSlot, String newLanguage ) {
	super();
	//System.out.println("i="+i+"  u="+u+"  tp="+tp+"  tt="+tt+"  lp="+lp+"  ll="+ll);
	owner = ts; 
	baseOwner = ts.agbase;
	baseInstance = ( com.franz.agbase.impl.ValueObjectImpl) com.franz.agbase.impl.AGFactory.makeLiteral(baseOwner, i, newLabel, 
			newTypeId, newType, newLangSlot, newLanguage);
	baseLiteral = (LiteralNodeImpl) baseInstance;
	}
    
    LiteralImpl ( AllegroGraph ts, com.franz.agbase.LiteralNode base ) {
    	owner = ts; baseOwner = ts.agbase;
    	baseInstance = (ValueObjectImpl) base;
    	baseLiteral = (LiteralNodeImpl) baseInstance;
    }

    public LiteralImpl() { 	}

	/**
     * Return the string associated with the Literal instance.
     * @return A string or null.
     * <p>
     * If the returned value is null, the string value is not in the local
     * Java cache, and must be retrieved from the AllegroGraph server with
     * a call to getLabel().
     */
    public String queryLabel () { return baseLiteral.queryLabel(); }


    /**
     * Return the string associated with the Literal instance.
     * @return A string.
     * If the value is not in the Java cache, retrieve it from the triple store.
     */
    public String getLabel() { return baseLiteral.getLabel(); }

    /**
     * Retrieve the string label of the datatype of the Literal.
     * @return null if the information is not in the local cache or
     *    if the Literal does not have a datatype label.
     * <p>
     * If the returned value is null, getType() or getDatatype() must be called
     * to get the actual value.
     */
    public String queryType () { return baseLiteral.queryType(); }

/**
 * Retrieve the datatype as a URI instance.
 * @return 
 * If the string label is not in the local Java cache, this method
 * requires a round-trip to the AllegroGraph server.
 */
    public URI getDatatype () { 
    	URINode dt = baseLiteral.getDatatype();
    	if ( dt==null ) return null;  // datatype can be null [bug18073]
    	return  (URI) CursorImpl.newValueNode(owner, dt);
    }
    

    /**
     * Retrieve the string label for the datatype of the Literal.
     * @return a string, or null if the Literal does not have a datatype field.
     * This operation may require a round-trip to the AllegroGraph triple store.
     */
    public String getType () { return baseLiteral.getType(); }

    /**
     * Retrieve the language field of the Literal.
     * @return null if the value is not in the local cache or
     *     if the Literal does not have a language label.
     * <p>
     * If the returned value is null, getLanguage() must be called
     * to get the actual value.
     */
    public String queryLanguage () { return baseLiteral.queryLanguage(); }


    /**
     * Retrieve the language qualifier of the Literal.
     * 
     * @return null if the Literal does not have a language qualifier.
     */
    public String getLanguage () { return baseLiteral.getLanguage();  }

  

    /**
     * This method overrides the generic toString method.
     * This method generates a more readable output string of the 
     * form "&lt;Literal <i>id</i>: <i>label</i>[langortype]&gt;".
     */
    public String toString() { return baseLiteral.toString(); } 
   
    
    /**
     * Implement equality for Literal instances.
     * <p>
     * Two Literal instances are equal if both are registered in the
     * AllegroGraph triple store and  they have identical
     * AllegroGraph part id numbers.
     * <p>
     * Otherwise, the string representations are compared.
     */
    public boolean equals(Object other) {
    	if ( other==null ) return false;
    	if ( this==other ) return true;
    	switch(sameAGId(other)) {
    	case 1: return true;
    	case 0: return false;
    	}
        if (other instanceof LiteralImpl) {
           // as per the OpenRDF model API:
           return toString().equals(other.toString());
        }
		return false;
    }

    /**
     * Compute the hashcode of a Literal instance.
     * <p>
     * The hashcode of a Literal instance is the hashcode
     * of its string representation.
     */
    public int hashCode() {
        // as per the OpenRDF model API:
        return toString().hashCode();
    }
    
    /**
     * Add this literal to the AllegroGraph triple store.
     * If the literal already is in the triple store, do nothing.
     * <p>
     * A Literal instance is in the triple store if queryAGId() returns
     * a non-null value.
     * @throws AllegroGraphException 
     *
     */
    public void add () throws AllegroGraphException {
    	if ( canReference() ) return;
    	baseLiteral.add();
    }
    
    public UPI getAGId () throws AllegroGraphException {
    	return baseLiteral.getAGId();
    }

}
