//***** BEGIN LICENSE BLOCK *****
//Version: MPL 1.1
//
//The contents of this file are subject to the Mozilla Public License Version
//1.1 (the "License"); you may not use this file except in compliance with
//the License. You may obtain a copy of the License at
//http://www.mozilla.org/MPL/
//
//Software distributed under the License is distributed on an "AS IS" basis,
//WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
//for the specific language governing rights and limitations under the
//License.
//
//The Original Code is the AllegroGraph Java Client interface.
//
//The Original Code was written by Franz Inc.
//Copyright (C) 2006 Franz Inc.  All Rights Reserved.
//
//***** END LICENSE BLOCK *****

package com.franz.ag.impl;

import org.openrdf.model.Statement;

import com.franz.ag.AllegroGraph;
import com.franz.ag.AllegroGraphException;
import com.franz.ag.Cursor;
import com.franz.ag.Triple;
import com.franz.ag.UPI;
import com.franz.ag.ValueNode;
import com.franz.ag.util.AGInner;
import com.franz.agbase.impl.TriplesIteratorImpl;
import com.franz.agbase.impl.UPIImpl;
import com.franz.agbase.util.AGBase;

/**
 * This class implements a generator for multiple Triple instances.
 * <p>
 * Many triple store search operations may generate an indeterminate number of
 * results. These operations return a Cursor instance which may be used to
 * iterate through the available results.
 * <p>
 * There are no public constructors.  Instances are created by search
 * operations.
 */
public class CursorImpl implements Cursor {
	
	public TriplesIteratorImpl baseInstance;
	/**
	 * The default number of triples to transfer from AG to Java whenever a
	 * Cursor is advanced. The built-in initial value is 1000.
	 */
	public static int defaultLookAhead = 1000;

	/**
	 * Query the default look-ahead value for this cursor instance.
	 * @return an integer
	 * @see #setDefaultLookAhead(int)
	 */
	public static int getDefaultLookAhead() {
		return defaultLookAhead;
	}

	/**
	 * Set the default look-ahead value for this cursor instance.
	 * @param lh an integer.  Any value less than 1 denotes the built-in default of 1000.
	 * @see #setLookAhead(int)
	 */
	public static void setDefaultLookAhead(int lh) {
		if (lh < 1)
			defaultLookAhead = 1000;
		else
			defaultLookAhead = lh;
	}

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#getLookAhead()
	 */
	public int getLookAhead() {
		return lookAhead;
	}

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#setLookAhead(int)
	 */
	public void setLookAhead(int lh) {
		if (lh < 1)
			lookAhead = defaultLookAhead;
		else
			lookAhead = lh;
	}

	/**
	 * Value returned when a triple part is not there.
	 */
	static final long NO_VALUE = -1;

	private AllegroGraph ag;
	public AllegroGraph getAG () { return ag; }

	public Object getSource () { return baseInstance.getSource(); }

	int lookAhead = defaultLookAhead;

	// NEW ACCESSOR HERE, INSTEAD OF CALLING PUBLIC FIELD DIRECTLY - RMM
	public AGBase getStore () {return this.ag;}

	public static final Cursor emptyCursor = new CursorImpl(null, null, null);
	
	public static Cursor emptyCursor () { return emptyCursor; }
	 
	static int initLookAhead ( AGInner rts ) {
		if ( rts==null ) return 0;
		if ( 0==rts.defaultLookAhead ) return defaultLookAhead;
		return rts.defaultLookAhead;
	}


//	 Use package access here because only use should be in AGFactory
	CursorImpl(AllegroGraph rts, Object cursor, UPIImpl[] newts) {
		super();
		com.franz.agbase.AllegroGraph rbase = null;
		if ( rts!=null ) rbase = rts.agbase;
		baseInstance = (TriplesIteratorImpl) 
		       com.franz.agbase.impl.AGFactory.makeCursor(rbase, cursor, newts);
		lookAhead = initLookAhead(rts);
		ag = rts;
	}

//	 Use package access here because only use should be in AGFactory
	CursorImpl(AllegroGraph rts, Object cursor, UPIImpl[] newts,
			String[] newdefs) {
		super();
		baseInstance = (TriplesIteratorImpl) com.franz.agbase.impl.AGFactory.makeCursor(rts.agbase, cursor, newts, newdefs);
		lookAhead = initLookAhead(rts);
		ag = rts;
	}
	
	CursorImpl ( AllegroGraph rts, com.franz.agbase.TriplesIterator base ) {
		super();
		baseInstance = (TriplesIteratorImpl) base;
		lookAhead = initLookAhead(rts);
		ag = rts;
	}


	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#get_id()
	 */
	public long get_id() { return baseInstance.get_id(); }

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#getS()
	 */
	public synchronized UPI getS() { return baseInstance.getS();  }

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#querySubject()
	 */
	public String querySubject() { return baseInstance.querySubject(); }

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#queryObject()
	 */
	public String queryObject() { return baseInstance.queryObject(); }

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#queryPredicate()
	 */
	public String queryPredicate() { return baseInstance.queryPredicate(); }

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#queryContext()
	 */
	public String queryContext() { return baseInstance.queryContext(); } // quad-store

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#getSubjectLabel()
	 */
	public synchronized String getSubjectLabel() throws AllegroGraphException {
		return baseInstance.getSubjectLabel();
	}

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#getSubject()
	 */
	public ValueNode getSubject() throws AllegroGraphException {
		if ( subject==null )
			subject = newValueNode(ag, baseInstance.getSubject());
		return subject;
	}

	
	private ValueNode context = null;
	private ValueNode subject = null;
	private ValueNode predicate = null;
	private ValueNode object = null;
	
	private void freshTriple () {
		context = null; subject = null; predicate = null; object = null;
	}
	
	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#getContext()
	 */
	public ValueNode getContext() throws AllegroGraphException {
		if ( context==null )
			context = newValueNode(ag, baseInstance.getContext());
		return context;
	}
	
	public static ValueNode newValueNode( AllegroGraph ag, com.franz.agbase.ValueNode base ) {
		if ( base instanceof com.franz.agbase.URINode )
			return AGFactory.makeNode(ag, (com.franz.agbase.URINode) base);
		if ( base instanceof com.franz.agbase.EncodedLiteral )
			return AGFactory.makeEncodedLiteral(ag, (com.franz.agbase.EncodedLiteral)base);
		if ( base instanceof com.franz.agbase.LiteralNode )
			return AGFactory.makeLiteral(ag, (com.franz.agbase.LiteralNode)base);
		if ( base instanceof com.franz.agbase.BlankNode )
			return AGFactory.makeBlankNode(ag, (com.franz.agbase.BlankNode)base);
		if ( base instanceof com.franz.agbase.DefaultGraph )
			return AGFactory.makeDefaultGraph(ag, (com.franz.agbase.DefaultGraph)base);
		throw new IllegalStateException("Unexpected base " + base);
	}

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#getP()
	 */
	public synchronized UPI getP() { return baseInstance.getP(); }

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#getPredicateLabel()
	 */
	public synchronized String getPredicateLabel() throws AllegroGraphException {
		return baseInstance.getPredicateLabel();
	}

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#getPredicate()
	 */
	public ValueNode getPredicate() throws AllegroGraphException {
		if ( predicate==null )
			predicate = newValueNode(ag, baseInstance.getPredicate());
		return predicate;
	}

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#getO()
	 */
	public synchronized UPI getO() { return baseInstance.getO(); }

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#getC()
	 */
	public synchronized UPI getC() { return baseInstance.getC(); }

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#getObjectLabel()
	 */
	public synchronized String getObjectLabel() throws AllegroGraphException {
		return baseInstance.getObjectLabel();
	}

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#getContextLabel()
	 */
	public synchronized String getContextLabel() throws AllegroGraphException {
		return baseInstance.getContextLabel();
	}

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#getObject()
	 */
	public ValueNode getObject() throws AllegroGraphException {
		if ( object==null )
			object = newValueNode(ag, baseInstance.getObject());
		return object;
	}

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#queryTriple()
	 */
	public synchronized TripleImpl queryTriple() {
		com.franz.agbase.impl.TripleImpl base = baseInstance.queryTriple();
		if ( base==null ) return null;
		return (TripleImpl) AGFactory.makeTriple(ag, base);
	}
	
	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#getTriple()
	 */
	public synchronized Triple getTriple() throws AllegroGraphException {
		com.franz.agbase.impl.TripleImpl base = (com.franz.agbase.impl.TripleImpl) baseInstance.getTriple();
		if ( base==null ) return null;
		
		return AGFactory.makeTriple(ag, base);
	}

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#next()
	 */
	public synchronized Statement next() {
		freshTriple();
		baseInstance.next();
		return queryTriple();
	}
	
	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#getNext()
	 */
	public synchronized Triple getNext() {
		freshTriple();
		baseInstance.getNext();
		return queryTriple();
	}

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#atTriple()
	 */
	public boolean atTriple() {
		return baseInstance.atTriple();
	}

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#hasNext()
	 */
	public boolean hasNext() {
		return baseInstance.hasNext();
	}

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#limitReached()
	 */
	public boolean limitReached() {
		return baseInstance.limitReached();
	}

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#close()
	 */
	public void close() {
		freshTriple();
		baseInstance.close();
	}

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#step()
	 */
	synchronized public boolean step() throws AllegroGraphException {
		freshTriple();
		return baseInstance.step();
	}


	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#step(int)
	 */
	synchronized public Triple[] step(int n) throws AllegroGraphException {
		freshTriple();
		com.franz.agbase.Triple[] baseTriples = baseInstance.step(n);
		Triple[] r = new Triple[baseTriples.length];
		for (int i = 0; i < baseTriples.length; i++) {
			r[i] = AGFactory.makeTriple(ag, baseTriples[i]);
		}
		return r;
	}

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#toString()
	 */
	public synchronized String toString() {
		return baseInstance.toString();
	}

	/* (non-Javadoc)
	 * @see com.franz.ag.Cursor#remove()
	 */
	public synchronized void remove() {
		baseInstance.remove();
	}

}
