package com.franz.agbase.examples;

import com.franz.agbase.*;

public class AGSNAGenerators {

	/**
	 * Demonstrates some basics of using generators in a Social Network.
	 * 
	 * @param args unused
	 * @throws AllegroGraphException
	 */
	public static void main(String[] args) throws AllegroGraphException {
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}

		// Create a fresh triple-store.
		AllegroGraph ts = ags.renew("snagenerators", AGPaths.TRIPLE_STORES);
		
		// Register namespaces
		ts.registerNamespace("ex", "http://example.org/");
		
		// Load a small graph of data
		AGLoadNtriples.loadNTriplesWithTiming(ts, AGPaths.dataSources("sna-doublediamond.nt"));
		
		// Get an SNA extension instance for this store to work with SNA features 
		SNAExtension sna = ts.getSNAExtension();
		
		// Register an objects-of generator
		Object[] parts = new Object[2];
		parts[0] = "objects-of";
		parts[1] = "!ex:to";
		sna.registerGenerator("to-objects", parts);
		
		// Register a subjects-of generator
		parts[0] = "subjects-of";
		parts[1] = "!ex:to";
		sna.registerGenerator("to-subjects", parts);
		
		// Register an undirected generator
		parts[0] = "undirected";
		parts[1] = "!ex:to";
		sna.registerGenerator("to-undirected", parts);
		
		// Define a group of actors
		String[] group = {"!ex:a","!ex:b","!ex:c","!ex:d","!ex:e","!ex:f","!ex:g","!ex:h"};
		
		// And the list of generators
		String[] generators = {"to-objects","to-subjects","to-undirected"};
		
		// Show the nodal degrees of each node with respect to each generator 
		for (int i=0;i<group.length;i++) {
			for (int j=0;j<generators.length;j++){
				long nodalDegree = sna.getNodalDegree(group[i], generators[j]);
				System.out.println("Nodal Degree of " + group[i] + " using " + generators[j] + " is " + nodalDegree);
			}
		}

		// Show the nodal neighbors of each node with respect to each generator
		for (int i=0;i<group.length;i++) {
			for (int j=0;j<generators.length;j++){
				UPI[] nodalNeighbors = sna.getNodalNeighbors(group[i], generators[j]);
				System.out.println("Nodal Neighbors of " + group[i] + " using " + generators[j] + " are " 
						+ AGUtils.upiArrayToString(ts,nodalNeighbors));
			}
		}

		// Close the store and disconnect from the server.
		ts.closeTripleStore();
		ags.disable();
	}

}
