package com.franz.ag;

import com.franz.ag.impl.AGFactory;
import com.franz.ag.util.AGInner;
import com.franz.agbase.impl.NamedAttributeList;
import com.franz.agbase.util.AGBase;
import com.franz.agbase.util.SPARQLQueryInternals;


/**
 * This class collects the parameters and results of a SPARQL query.
 * <p>
 * The typical sequence of steps is to
 *  <ul>
 *   <li>Create an instance.
 *   <li>Set or modify some of the optional parameters.
 *   <li>Call one of the action methods
 *        <ul>
 *          <li>{@link #ask()}
 *          <li>{@link #construct()}
 *          <li>{@link #count()}
 *          <li>{@link #describe()}
 *          <li>{@link #run()}
 *          <li>{@link #select()}
 *         </ul>
 *    <li>Use the value returned by the action method, or extract the result from the
 *        instance.
 *   </ul>
 *  The choice of action method is determined by the kind of query and by the desired
 *  datatype of the result.
 * <p>
 * This document is not intended to be a SPARQL tutorial or reference.
 * Please refer to other documents for the definition of SPARQL syntax and semantics
 * and for the precise behavior of the SPARQL engine.
 * 
 * @author mm
 *
 */
public class SPARQLQuery 
extends SPARQLQueryInternals
implements SPARQLQueryConstants {
	
	/**
	 * Create a new empty SPARQL query with default arguments.
	 *
	 */
	public SPARQLQuery () {
		super();
		queryAttributes = new NamedAttributeList(queryOptions);
		baseInstance = new com.franz.agbase.SPARQLQuery();
		}
	
	private com.franz.agbase.SPARQLQuery baseInstance = null;
	
	/**
	 * Get the query string.
	 * @return the query string or null if the instance is not initialized/
	 */
	public String getQuery () { return baseInstance.getQuery(); }
	
	/**
	 * Set the query string.
	 * Setting the query string clears out any previous results.
	 * @param newQuery a string containing a complete well-formed SPARQL query.
	 */
	public void setQuery ( String newQuery ) {
		freshState();
		baseInstance.setQuery(newQuery);
		}
	

	private Cursor resultCursor = null;
	private ValueObject[][] resultArrays = null;
	private ValueObject[] resultArray = null;
	private AllegroGraph ag = null;
	
	private void freshState() {
		freshStateVars();
		resultCursor = null;
		resultArrays = null;
		resultArray = null;
	}
	
	/**
	 * Query the result of a query that has a boolean result.
	 * @return the boolean result.
	 * @throws IllegalStateException if a boolean result is not available.
	 */
	public boolean getBooleanResult () { return baseInstance.getBooleanResult(); }
	
	/**
	 * Query the count value of a query that returns a numeric result.
	 * @return the number of results returned by a count() call.
	 * @throws IllegalStateException if the count is not available.
	 */
	public long getResultCount () { return baseInstance.getResultCount(); }
	
	/**
	 * Query the names and order of the variables in a SELECT query result.
	 * @return An array of strings.
	 * @throws IllegalStateException if the names are not available.
	 */
	public String[] getResultNames () { 
		// NOT DELGATED to agbase because result arays is built locally
		// and result names are kept locally in AllegroGraph class.
		if ( ag==null ) throw new IllegalStateException("Not linked to AllegroGraph instance.");
		if ( resultArrays!=null ) 
			return ag.valueNames(resultArrays);
		if ( resultArray!=null )
			return ag.valueNames(resultArray);
		throw new IllegalStateException("There is no array of results.");
	}
	
	
	/**
	 * Query the includeInferred option.
	 * @return the includeInferred
	 */
	public boolean isIncludeInferred() { return baseInstance.isIncludeInferred(); }
	
	/**
	 * Modify the includeInferred option.
	 * @param includeInferred the desired value.
	 */
	public void setIncludeInferred(boolean includeInferred) {
		freshState();
		baseInstance.setIncludeInferred(includeInferred);
	}
	
	/**
	 * Query the hasValue option.
	 * When true, hasValue reasoning is enablede.
	 * @return 1 if true, 0 if false, -1 if unspecified (ie server default)
	 */
	public int getHasValue () { return baseInstance.getHasValue(); }
	
	/**
	 * Enable or disable hasValue reasoning.
	 * This option is in effect only if reasoning is enabled.
	 * @param v true or false
	 */
	public void setHasValue ( boolean v ) {
		freshState();
		baseInstance.setHasValue(v);
	}
	
	/**
	 * Query the limit option.
	 * The limit option overrides a LIMIT clause in the query.
	 * @return the limit
	 */
	public long getLimit() { return baseInstance.getLimit(); }
	
	/** Set the limit option.
	 * The limit option overrides a LIMIT clause in the query.
	 * @param limit the limit to set.  A zero or negative value is ignored.
	 */
	public void setLimit( int limit ) {
		freshState();
		baseInstance.setLimit(limit);
	}
	
	/**
	 * Query the offset option.
	 * The offset option overrides an OFFSET clause in the query.
	 * @return the offset. A zero or negative value is ignored.
	 */
	public long getOffset() { return baseInstance.getOffset(); }
	
	/**
	 * Set the offset option.
	 * The offset option overrides an OFFSET clause in the query.
	 * @param offset the offset to set.  A zero or negative value is ignored.
	 */
	public void setOffset( int offset ) {
		freshState();
		baseInstance.setOffset(offset);
	}
	
	/**
	 * Query the RDFFormat option.
	 * This option applies to queries that return an RDF XML serialization string.
	 * @return the RDFFormat
	 */
	public String getRDFFormat() { return baseInstance.getRDFFormat(); }
	
	/**
	 * Set the RDFFormat option.
	 * This option applies to queries that return an RDF XML serialization string.
	 * @param format the RDFFormat to set
	 */
	public void setRDFFormat(String format) {
		freshState();
		baseInstance.setRDFFormat(format);
	}
	
	/**
	 * Query the resultsFormat option.
	 * This option applies to queries that return a value set.
	 * @return the resultsFormat
	 */
	public String getResultsFormat() { return baseInstance.getResultsFormat(); }
	
	/**
	 * Set the resultsFormat option.
	 * This option applies to queries that return a value set.
	 * @param resultsFormat the resultsFormat to set
	 */
	public void setResultsFormat(String resultsFormat) {
		freshState();
		baseInstance.setResultsFormat(resultsFormat);
	}
	/**
	 * Query the vars option.
	 * This option determines the content of the results array.
	 * @return the vars. A null value specifies the default result set which consists
	 * of the variables listed in the query, in the order listed in the query.
	 */
	public String getVars() { return baseInstance.getVars(); }
	
	/**
	 * Set the vars option.
	 * This option determines the content of the results array.
	 * @param vars null or a string containing variable names separated by spaces
	 */
	public void setVars(String vars) {
		freshState();
		baseInstance.setVars(vars);
	}
	
	
	
	/**
	 * Query the extended option.
	 * When true, extended SPARQL verbs are available.
	 * @return 1 if true, 0 if false, -1 if unspecified (ie server default)
	 */
	public int getExtended() { return baseInstance.getExtended(); }
	
	/**
	 * Set the extended option.
	 * When true, extended SPARQL verbs are available.
	 * @param extended the extended to set
	 */
	public void setExtended(boolean extended) {
		freshState();
		baseInstance.setExtended(extended);
	}
	
	/**
	 * Query the memoize option.
	 * When true, query results are memoized for potential time savings.
	 * @return 1 if true, 0 if false, -1 if unspecified (ie server default)
	 */
	public int getMemoized() { return baseInstance.getMemoized(); }
	
	/**
	 * Set the memoize option.
	 * When true, query results are memoized for potential time savings.
	 * @param memoized the memoized to set
	 */
	public void setMemoized(boolean memoized) {
		freshState();
		baseInstance.setMemoized(memoized);
	}
	
	/**
	 * Set the name of a memo table.
	 * The scope of the name is one AllegroGraphConnection instance.
	 * @param name
	 */
	public void setMemoTable ( String name ) {
		freshState();
		baseInstance.setMemoTable(name);
	}
	
	/**
	 * Query the name of the memo table.
	 * @return the name of the memo table, or null if none is set.
	 */
	public String getMemoTable () {
		return baseInstance.getMemoTable();
	}
	
	/**
	 * Set the name of a load-function option.
	 * @param name
	 */
	public void setLoadFunction ( String name ) {
		freshState();
		baseInstance.setLoadFunction(name);
	}
	
	/**
	 * Query the name of the load-function.
	 * @return the name of the load-function, or null if none is set.
	 */
	public String getLoadFunction () {
		return baseInstance.getLoadFunction();
	}
	
	/**
	 * Set the default-prefixes option.
	 * @param prefixes an array of alternating prefix and URI strings.
	 */
	public void setDefaultPrefixes ( String[] prefixes ) {
		freshState();
		baseInstance.setDefaultPrefixes(prefixes);
	}
	
	/**
	 * Set the default-prefixes option.
	 * @param prefixes a NamespaceRegistry instance that defines the desired prefixes.
	 */
	public void setDefaultPrefixes ( NamespaceRegistry prefixes ) {
		freshState();
		baseInstance.setDefaultPrefixes(prefixes);
	}
	
	/**
	 * Query the default-prefixes option.
	 * @return an array of strings. The content of the array is a list 
	 *    of alternating prefix and URI values.
	 */
	public String[] getDefaultPrefixes () {
		return baseInstance.getDefaultPrefixes();
	}
	
	/**
	 * Set the default-base option for this SPARQL query.
	 * @param base a string containing the default base URI.
	 */
	public void setDefaultBase ( String base ) {
		freshState();
		baseInstance.setDefaultBase(base);
	}
	
	/**
	 * Query the default base value of the query.
	 * @return null if the sever default applies, or the steing value set by the user.
	 */
	public String getDefaultBase () {
		return baseInstance.getDefaultBase();
	}
	
	/**
	 * Set the from-named option,
	 * @param uriLabels an array of URI strings
	 */
	public void setFromNamed ( String[] uriLabels ) {
		freshState();
		baseInstance.setFromNamed(uriLabels);
	}
	
	/**
	 * Query the from-named option.
	 * @return an array of strings containing the from-named URIs.
	 */
	public String[] getFromNamed () {
		return baseInstance.getFromNamed();
	}
	
	/**
	 * Set the from option,
	 * @param uriLabels an array of URI strings
	 */
	public void setFrom ( String[] uriLabels ) {
		freshState();
		baseInstance.setFrom(uriLabels);
	}
	
	/**
	 * Query the from option.
	 * @return an array of strings containing the from URIs.
	 */
	public String[] getFrom () {
		return baseInstance.getFrom();
	}
	
	/**
	 * Set the default-data-set-behavior option to "all".
	 *
	 */
	public void setDefaultDatasetBehaviorAll () {
		freshState();
		baseInstance.setDefaultDatasetBehaviorAll();
	}
	
	/**
	 * Set the default-data-set-behavior option to "default".
	 *
	 */
	public void setDefaultDatasetBehaviorDefault () {
		freshState();
		baseInstance.setDefaultDatasetBehaviorDefault();
	}
	
	/**
	 * Query the default dataset behavior for this SPARQL query.
	 * @return null if the server default applies, or the string "all" or "default".
	 */
	public String getDefaultDatasetBehavior () {
		return baseInstance.getDefaultDatasetBehavior();
	}
	
	/**
	 * Query the engine that was or will be used for the SPARQL query.
	 * @return 
	 */
	public ENGINE getEngine () { return baseInstance.getEngine(); }
	
	/**
	 * Set the engine that will be used for the SPARQL query.
	 * @param e
	 */
	public void setEngine ( ENGINE e ) {
		freshState();
		baseInstance.setEngine(e);
	}
	
	/**
	 * Query the engine that was or will be used for the SPARQL query.
	 * @return 
	 */
	public PLANNER getPlanner () { return baseInstance.getPlanner(); }
	
	/**
	 * Set the engine that will be used for the SPARQL query.
	 * @param e
	 */
	public void setPlanner ( PLANNER e ) {
		freshState();
		baseInstance.setPlanner(e);
	}
	
	
	/**
	 * Query the with-variables option.
	 * @return a copy of the array that was used to set this option.
	 */
	public Object[] getWithVariables() {
		return  saveWithVars;
	}
	
	//private Object[] saveWithVars = null;
	/**
	 * Set the with-variables option for the SPARQL query.
	 * @param ag the AllegroGraph instance where the variable values are resolved.
	 * @param withVariables an array of alternating variable names and values.
	 *      The variable names must be strings. 
	 *      The variable values may be any valid triple part specifier as in addStatement.
	 */
	public void setWithVariables( AllegroGraph ag, Object[] withVariables ) {
		String[] withVars = saveWithVariables(ag, withVariables);
		freshState();
		baseInstance.queryAttributes.setAttribute(WITHV, withVars);
	}
	protected String convertWithVariable( AGBase ag, Object var ) {
		return ((AllegroGraph) ag).refToString(var);
	}
	
	/**
	 * Query the results of a SPARQL query that returns a one-dimensional 
	 * array of results. 
	 * This result is available only after a call to one of the selectSingle()
	 * methods.
	 * @return the resultArray
	 * @throws IllegalStateException if the array is not available.
	 */
	public ValueObject[] getResultArray() {
		if ( null!=resultArray ) return resultArray;
		throw new IllegalStateException("ResultArray is not set.");
	}
	
	/**
	 * Query the results of a SPARQL query that returns an array of result arrays.
	 * This result is available only after a call to one of the select()
	 * methods.
	 * @return the resultArrays
	 */
	public ValueObject[][] getResultArrays() {
		if ( null!=resultArrays ) return resultArrays;
		throw new IllegalStateException("ResultArrays is not set.");
	}
	
	/**
	 * Query the result of a SPARQL query that returns a collection of triples.
	 * This result is available only after a call to one of the describe()
	 * or construct()
	 * methods.
	 * @return the resultCursor
	 */
	public Cursor getResultCursor() {
		if ( haveResultCursor ) return resultCursor;
		throw new IllegalStateException("ResultCursor is not set.");
	}
	
	/**
	 * Query the result of a SPARQL query that returns a string result.
	 * This result is available only after a call to one of the run()
	 * methods.
	 * @return the resultString
	 */
	public String getResultString() {
		if ( null!=resultString ) return resultString;
		throw new IllegalStateException("ResultString is not set.");
	}

	private void validate ( AllegroGraph ag ) {
		freshState();
		if ( ag!=null ) this.ag = ag;
		if ( this.ag==null )
			throw new IllegalStateException("Cannot run without a triple store.");
	}
	
	/**
	 * Specify the triple store agaist which this SPARQL query will run.
	 * Setting the store clears out any previous results.
	 * @param ag
	 */
	public void setTripleStore ( AllegroGraph ag ) {
		freshState();
		this.ag = ag;
	}
	
	/**
	 * Query the triple store agaist which this SPARQL query has or will run.
	 * @return the AllegroGraoh instance or null.
	 */
	public AGInner getTripleStore () { return ag; }
	
	/**
	 * Run a SPARQL query that retrieves a set of variable bindings.
	 * The query must be a SPARQL SELECT query.
	 * @return an array of result sets.  Each result set is an array of values.
	 *    The first array of results may be incomplete.
	 *    Additional results are obtained by calling {@link AllegroGraph#selectMore(ValueObject[][])}.
	 * @throws AllegroGraphException
	 */
	public ValueObject[][] select () throws AllegroGraphException {
		validate(null);
		resultArrays = ag.twinqlSelect(baseInstance.isIncludeInferred(),
				baseInstance.getQuery(), baseInstance.getVars(),
				(int)(baseInstance.getLimit()), 
				(int)(baseInstance.getOffset()),
				baseInstance.queryAttributes.getList());
		resultVars = ag.valueNames(resultArrays);
		return resultArrays;
	}
	
	/**
	 * Run a SPARQL query that retrieves a set of variable bindings.
	 * @param ag the AllogroGraph instance against which the query will run. 
	 * @return an array of result sets.  Each result set is an array of values.
	 *     The first array of results may be incomplete.
	 *    Additional results are obtained by calling {@link AllegroGraph#selectMore(ValueObject[][])}.
	 * @throws AllegroGraphException
	 */
	public ValueObject[][] select ( AllegroGraph ag ) throws AllegroGraphException {
		this.ag = ag;
		return select();
	}
	
	/**
	 * Run a SPARQL query that retrieves a set of variable bindings.
	 * @param query the complete and well-formed SPARQL SELECT query string.
	 * @return an array of result sets.  Each result set is an array of values.
	 *    The first array of results may be incomplete.
	 *    Additional results are obtained by calling {@link AllegroGraph#selectMore(ValueObject[][])}.
	 * @throws AllegroGraphException
	 */
	public ValueObject[][] select ( String query ) throws AllegroGraphException {
		baseInstance.setQuery(query);
		return select();
	}
	
	public ValueObject[][] select ( AllegroGraph ag, String query ) throws AllegroGraphException {
		this.ag = ag;
		baseInstance.setQuery(query);
		return select();
	}
	
	/**
	 * Run a SPARQL query that retrieves all the bindings of a single variable.
	 * The query must be a SPARQL SELECT query that returns a single value in each 
	 * result set.
	 * @return an array of results.
	 *    The first array of results may be incomplete.
	 *    Additional results are obtained by calling {@link AllegroGraph#selectMore(ValueObject[])}.
	 * @throws AllegroGraphException
	 */
	public ValueObject[] selectSingle () throws AllegroGraphException {
		validate(null);		
		Object[] v = ag.verifyEnabled().twinqlSelect(ag, 
				baseInstance.getQuery(), 
				baseInstance.getVars(),
				(int)(baseInstance.getLimit()), 
				(int)(baseInstance.getOffset()),
				ag.selectLimit,
				baseInstance.isIncludeInferred(),
				baseInstance.queryAttributes.getList());
		resultArray = (ValueObject[]) ag.selectValuesArray(true, v, true);
		resultVars = ag.valueNames(resultArray);
		return resultArray;
	}
	public ValueObject[] selectSingle ( AllegroGraph ag, String query ) throws AllegroGraphException {
		this.ag = ag;
		baseInstance.setQuery(query);
		return selectSingle();
	}
	public ValueObject[] selectSingle ( String query ) throws AllegroGraphException {
		baseInstance.setQuery(query);
		return selectSingle();
	}
	public ValueObject[] selectSingle ( AllegroGraph ag ) throws AllegroGraphException {
		this.ag = ag;
		return selectSingle();
	}
	
	
	/**
	 * Run a SPARQL query that retrieves a set (or row) of variable bindings.
	 * The query must be a SPARQL SELECT query.
	 * @return the number of result sets found.  The actual result sets are discarded.
	 * <p>
	 * The result may also be obtained subsequently by calling {@link #getResultCount()}.
	 * @throws AllegroGraphException
	 */
	public long count () throws AllegroGraphException {
		validate(null);
		return baseInstance.count(ag.agbase);
	}
	
	/**
	 * Run a SPARQL query that retrieves a set (or row) of variable bindings.
	 * @param ag the AllegroGraph instance where the query will run.
	 *     This instance is remembered as if set with {@link #setTripleStore(AllegroGraph)}.
	 * @return the number of result sets found.
	 * <p>
	 * @see #count().
	 * @throws AllegroGraphException
	 */
	public long count ( AllegroGraph ag ) throws AllegroGraphException {
		this.ag = ag;
		return count();
	}
	
	/**
	 * Run a SPARQL query that retrieves a set (or row) of variable bindings.
	 * @param ag the AllegroGraph instance where the query will run.
	 *     This instance is remembered as if set with {@link #setTripleStore(AllegroGraph)}.
	 * @param query a string containing a well-formed SPARQL SELECT query.
	 *     The query is remembered as if set with {@link #setQuery(String)}.
	 * @return the number of result sets found.
	 * <p>
	 * @see #count().
	 * @throws AllegroGraphException
	 */
	public long count ( AllegroGraph ag, String query ) throws AllegroGraphException {
		this.ag = ag;
		baseInstance.setQuery(query);
		return count();
	}
	
	/**
	 * Run a SPARQL query that retrieves a set (or row) of variable bindings.
	 * @param query a string containing a well-formed SPARQL SELECT query.
	 *     The query is remembered as if set with {@link #setQuery(String)}.
	 * @return the number of result sets found.
	 * <p>
	 * @see #count().
	 * @throws AllegroGraphException
	 */
	public long count ( String query ) throws AllegroGraphException {
		baseInstance.setQuery(query);
		return count();
	}
	
	
	/**
	 * Run a SPARQL query that returns a boolean result.
	 * The query must be a SPARQL ASK query.
	 * @return true if the query succeeded.
	 * <p>
	 * The result may also be obtained subsequently by calling {@link #getBooleanResult()}.
	 * @throws AllegroGraphException
	 */
	public boolean ask () throws AllegroGraphException {
		validate(null);
		return baseInstance.ask(ag.agbase);
	}
	
	/**
	 * Run a SPARQL query that returns a boolean result.
	 * @param ag the AllegroGraph instance where the query will run.
	 *     This instance is remembered as if set with {@link #setTripleStore(AllegroGraph)}.
	 * @return true if the query succeeded.
	 * 
	 * @see #ask().
	 * 
	 * @throws AllegroGraphException
	 */
	public boolean ask ( AllegroGraph ag ) throws AllegroGraphException {
		this.ag = ag;
		return ask();
	}
	
	/**
	 * Run a SPARQL query that returns a boolean result.
	 * @param ag the AllegroGraph instance where the query will run.
	 *     This instance is remembered as if set with {@link #setTripleStore(AllegroGraph)}.
	 * @param query a string containing a well-formed SPARQL ASK query.
	 *     The query is remembered as if set with {@link #setQuery(String)}.
	 * @return true if the query succeeded.
	 * 
	 * @see #ask().
	 * 
	 * @throws AllegroGraphException
	 */
	public boolean ask ( AllegroGraph ag, String query ) throws AllegroGraphException {
		this.ag = ag;
		baseInstance.setQuery(query);
		return ask();
	}
	
	/**
	 * Run a SPARQL query that returns a boolean result.
	 * @param query a string containing a well-formed SPARQL ASK query.
	 *     The query is remembered as if set with {@link #setQuery(String)}.
	 * @return true if the query succeeded.
	 * 
	 * @see #ask().
	 * 
	 * @throws AllegroGraphException
	 */
	public boolean ask ( String query ) throws AllegroGraphException {
		baseInstance.setQuery(query);
		return ask();
	}
	
	/**
	 * Run a SPARQL query that retrieves a set of statements.
	 * The query must be a SPARQL DESCRIBE query.
	 * @return a Cursor instance that will iterate over the resulting statements.
	 * <p>
	 * The result may also be obtained subsequently by calling {@link #getResultCursor()}.
	 * @throws AllegroGraphException
	 */
	public Cursor describe () throws AllegroGraphException {
		validate(null);
		resultCursor = agCursor(baseInstance.describe(ag.agbase));
		haveResultCursor = true;
		return resultCursor;
	}
	
	private Cursor agCursor ( com.franz.agbase.TriplesIterator base ) {
		return AGFactory.makeCursor(ag, base);
	}
	
	/**
	 * Run a SPARQL query that retrieves a set of statements.
	 * @param ag the AllegroGraph instance where the query will run.
	 *     This instance is remembered as if set with {@link #setTripleStore(AllegroGraph)}.
	 * @return a Cursor instance that will iterate over the resulting statements.
	 * <p>
	 * @see #describe().
	 * @throws AllegroGraphException
	 */
	public Cursor describe ( AllegroGraph ag ) throws AllegroGraphException {
		this.ag = ag;
		return describe();
	}
	
	/**
	 * Run a SPARQL query that retrieves a set of statements.
	 * @param ag the AllegroGraph instance where the query will run.
	 *     This instance is remembered as if set with {@link #setTripleStore(AllegroGraph)}.
	 * @param query a string containing a well-formed SPARQL DESCRIBE query.
	 *     The query is remembered as if set with {@link #setQuery(String)}.
	 * @return a Cursor instance that will iterate over the resulting statements.
	 * <p>
	 * @see #describe().
	 * @throws AllegroGraphException
	 */
	public Cursor describe ( AllegroGraph ag, String query ) throws AllegroGraphException {
		this.ag = ag;
		baseInstance.setQuery(query);
		return describe();
	}
	
	/**
	 * Run a SPARQL query that retrieves a set of statements.
	 * @param query a string containing a well-formed SPARQL DESCRIBE query.
	 *     The query is remembered as if set with {@link #setQuery(String)}.
	 * @return a Cursor instance that will iterate over the resulting statements.
	 * <p>
	 * @see #describe().
	 * @throws AllegroGraphException
	 */
	public Cursor describe ( String query ) throws AllegroGraphException {
		baseInstance.setQuery(query);
		return describe();
	}
	
	/**
	 * Run a SPARQL query that retrieves a set of statements.
	 * The query must be a SPARQL CONSTRUCT query.
	 * @return a Cursor instance that will iterate over the resulting statements.
	 * <p>
	 * The result may also be obtained subsequently by calling {@link #getResultCursor()}.
	 * @throws AllegroGraphException
	 */
	public Cursor construct () throws AllegroGraphException {
		validate(null);
		resultCursor = agCursor(baseInstance.construct(ag.agbase));
		haveResultCursor = true;
		return resultCursor;
	}
	
	/**
	 * Run a SPARQL query that retrieves a set of statements.
	 * @param ag the AllegroGraph instance where the query will run.
	 *     This instance is remembered as if set with {@link #setTripleStore(AllegroGraph)}.
	 * @return a Cursor instance that will iterate over the resulting statements.
	 * <p>
	 * @see #construct().
	 * @throws AllegroGraphException
	 */
	public Cursor construct ( AllegroGraph ag ) throws AllegroGraphException {
		this.ag = ag;
		return construct();
	}
	
	/**
	 * Run a SPARQL query that retrieves a set of statements.
	 * @param ag the AllegroGraph instance where the query will run.
	 *     This instance is remembered as if set with {@link #setTripleStore(AllegroGraph)}.
	 * @param query a string containing a well-formed SPARQL CONSTRUCT query.
	 *     The query is remembered as if set with {@link #setQuery(String)}.
	 * @return a Cursor instance that will iterate over the resulting statements.
	 * <p>
	 * @see #construct().
	 * @throws AllegroGraphException
	 */
	public Cursor construct ( AllegroGraph ag, String query ) throws AllegroGraphException {
		this.ag = ag;
		baseInstance.setQuery(query);
		return construct();
	}
	
	/**
	 * Run a SPARQL query that retrieves a set of statements.
	 * @param query a string containing a well-formed SPARQL CONSTRUCT query.
	 *     The query is remembered as if set with {@link #setQuery(String)}.
	 * @return a Cursor instance that will iterate over the resulting statements.
	 * <p>
	 * @see #construct().
	 * @throws AllegroGraphException
	 */
	public Cursor construct ( String query ) throws AllegroGraphException {
		baseInstance.setQuery(query);
		return construct();
	}
	
	/**
	 * Run a SPARQL query that returns a serialized string result.
	 * @return the string containing the serialized result.
	 * @throws AllegroGraphException if a problem was encountered during the search.
	 * @throws IllegalArgumentException if this instance is not properly initialized.
	 */
	public String run () throws AllegroGraphException {
		validate(null);
		return baseInstance.run(ag.agbase);
	}
	public String run ( AllegroGraph ag ) throws AllegroGraphException {
		this.ag = ag;
		return run();
	}
	public String run ( AllegroGraph ag, String query ) throws AllegroGraphException {
		this.ag = ag;
		baseInstance.setQuery(query);
		return run();
	}
	public String run ( String query ) throws AllegroGraphException {
		baseInstance.setQuery(query);
		return run();
	}
	
	
	
}
