package com.franz.ag;

import com.franz.ag.impl.AGFactory;
import com.franz.ag.impl.ValueObjectImpl;
import com.franz.ag.util.AGInner;
import com.franz.agbase.TriplesIterator;


/**
 * This class collects the parameters and results of a query that retrieves triples
 *  by patterns of triple parts.
 * <p>
 * A triple is located by a pattern of four components, the subject, predicate, object,
 * and graph.  Each of these may be specified as a string in NTriples notation,
 * a Value instance, a UPI instance.  A null value is a wildcard that matches anything.
 * The graph position can also be the empty string to denote the null context
 * or default graph of the triple store.
 * <p>
 * The inital value of the subject predicate and object positions is null.
 * The initial value of the graph position is the empty string.
 * <p>
 * Each of the triple components can also match a range of values if the 
 * corresponding end parameter is set.  In order to specify a range, the start 
 * and end must be specified as an EncodedLiteral instance or a UPI instance
 * that denotes an encoded UPI.  The strings "min" and "max" can also be used
 * to denote the minimum and maximum values of the corresponding UPI type.
 * <p>
 * The search can be managed in more detail by setting various attributes that
 * control what triples are examined and what triples are returned. 
 * <p>
 * An instance can be used many times to repeat the same query with identical
 * or different parameters.  The result fields are reset every time a parameter 
 * is modified.
 * 
 * 
 * @author mm
 *
 */
public class TriplesQuery {
	
	/**
	 * @return the context component of the query.
	 */
	public Object getContext() {
		return context;
	}

	/**
	 * Set the context component of the query.
	 * @param context the context to set.
	 *   A null valueis a wild card that matches anything.
	 *   The empty string denote the default context, or null graph.
	 * @throws AllegroGraphException 
	 */
	public void setContext(Object context) throws AllegroGraphException {
		freshState();
		baseInstance.setContext(mapToBase(context));
		this.context = context;
	}
	
	private Object mapToBase ( Object x ) {
		if ( x==null ) return x;
		if ( x instanceof ValueObjectImpl ) return ((ValueObjectImpl)x).baseInstance;
		return x;  // assume UPI or String (or error)
	}

	/**
	 * @return the contextEnd component of the query.
	 */
	public Object getContextEnd() {
		return contextEnd;
	}

	/**
	 * @param contextEnd the contextEnd to set
	 * @throws AllegroGraphException 
	 */
	public void setContextEnd(Object contextEnd) throws AllegroGraphException {
		freshState();
		baseInstance.setContextEnd(mapToBase(contextEnd));
		this.contextEnd = contextEnd;
	}

	/**
	 * @return the object component of the query.
	 */
	public Object getObject() {
		return object;
	}

	/**
	 * @param object the object to set
	 */
	public void setObject(Object object) {
		freshState();
		baseInstance.setObject(mapToBase(object));
		this.object = object;
	}

	/**
	 * @return the objectEnd component of the query.
	 */
	public Object getObjectEnd() {
		return objectEnd;
	}

	/**
	 * @param objectEnd the objectEnd to set
	 */
	public void setObjectEnd(Object objectEnd) {
		freshState();
		baseInstance.setObjectEnd(mapToBase(objectEnd));
		this.objectEnd = objectEnd;
	}

	/**
	 * @return the predicate component of the query.
	 */
	public Object getPredicate() {
		return predicate;
	}

	/**
	 * @param predicate the predicate to set
	 */
	public void setPredicate(Object predicate) {
		freshState();
		baseInstance.setPredicate(mapToBase(predicate));
		this.predicate = predicate;
	}

	/**
	 * @return the predicateEnd component of the query.
	 */
	public Object getPredicateEnd() {
		return predicateEnd;
	}

	/**
	 * @param predicateEnd the predicateEnd to set
	 */
	public void setPredicateEnd(Object predicateEnd) {
		freshState();
		baseInstance.setPredicateEnd(mapToBase(predicateEnd));
		this.predicateEnd = predicateEnd;
	}

	/**
	 * @return the subject component of the query.
	 */
	public Object getSubject() {
		return subject;
	}

	/**
	 * @param subject the subject to set
	 */
	public void setSubject(Object subject) {
		freshState();
		baseInstance.setSubject(mapToBase(subject));
		this.subject = subject;
	}

	/**
	 * @return the subjectEnd component of the query.
	 */
	public Object getSubjectEnd() {
		return subjectEnd;
	}

	/**
	 * @param subjectEnd the subjectEnd to set
	 */
	public void setSubjectEnd(Object subjectEnd) {
		freshState();
		baseInstance.setSubjectEnd(mapToBase(subjectEnd));
		this.subjectEnd = subjectEnd;
	}

	/**
	 * Create a new empty query with default arguments.
	 *
	 */
	public TriplesQuery () {
		baseInstance = new com.franz.agbase.TriplesQuery();
		}
	
	private com.franz.agbase.TriplesQuery baseInstance = null;
	
	// These are the values set by the user, to be returned if queried.
	private Object subject = null;
	private Object predicate = null;
	private Object object = null;
	private Object context = "";
	private Object subjectEnd = null;
	private Object predicateEnd = null;
	private Object objectEnd = null;
	private Object contextEnd = null;
	
	private boolean haveResultCursor = false;
	private Cursor resultCursor = null;
	private long resultCount = -1;
	private boolean haveBooleanResult = false;
	private boolean booleanResult = false;
	private AllegroGraph ag = null;
	private Triple resultTriple = null;
	
	private void freshState() {
		haveResultCursor = false;
		resultCursor = null;
		resultCount = -1;
		haveBooleanResult = false;
		booleanResult = false;
		resultTriple = null;
	}
	
	/**
	 * Query the result of a query that had a boolean result.
	 * @return the boolean result.
	 * @throws IllegalStateException if a boolean result is not available.
	 */
	public boolean getBooleanResult () { 
		if ( haveBooleanResult ) return booleanResult;
		throw new IllegalStateException("BooleanResult is not set.");
	}
	
	/**
	 * Query the count value of a query that returns a numeric result.
	 * @return the number of results returned by a count() call.
	 * @throws IllegalStateException if the count is not available.
	 */
	public long getResultCount () { 
		if ( -1<resultCount ) return resultCount;
		throw new IllegalStateException("ResultCount is not set.");
	}
	
	
	/**
	 * Query the includeInferred option.
	 * @return the includeInferred
	 */
	public boolean isIncludeInferred() {
		return baseInstance.isIncludeInferred();
	}
	
	/**
	 * Modify the includeInferred option.
	 * @param includeInferred the desired value.
	 */
	public void setIncludeInferred(boolean includeInferred) {
		freshState();
		baseInstance.setIncludeInferred(includeInferred);
	}
	
	
	/**
	 * Query the includeDeleted option.
	 * When true, deleted triples are included in the result.
	 * @return 1 if true, 0 if false, -1 if unspecified (ie server default)
	 */
	public int getIncludeDeleted() {
		return baseInstance.getIncludeDeleted();
	}
	
	/**
	 * Set the includeDeleted option.
	 * When true, deleted triples are included in the result.
	 * @param v true or false
	 */
	public void setIncludeDeleted(boolean v) {
		freshState();
		baseInstance.setIncludeDeleted(v);
	}
	
	/**
	 * Set the omitEncoded option.
	 * When true, encoded triples are not included in the search result.
	 * @param v true or false
	 */
	public void setOmitEncoded ( boolean v ) {
		freshState();
		baseInstance.setOmitEncoded(v);
	}
	
	/**
	 * Query the omitEncoded option.
	 * When true, encoded triples are not included in the search result.
	 * @return 1 if true, 0 if false, -1 if unspecified (ie server default)
	 */
	public int getOmitEncoded () {
		return baseInstance.getOmitEncoded();
	}
	
	/**
	 * Set the lookahead parameter for this query.
	 * The lookahead parameter specifies how many results are cached 
	 * in the client.  A large value allows quick access to results but may incur 
	 * a delay when the results are transmitted immediately after the query.
	 * A small value implies more round-trips to the server.
	 * <p>
	 * A zero value specifies that all the search results remain on the server.
	 * This option creates a Cursor that can be passed to a serializer in the server.
	 * @param lh the desired lookahead value.  The initial setting is 1000.
	 */
	public void setLookahead ( int lh ) {
		baseInstance.setLookahead(lh);
	}
	
	/**
	 * Query the lookahead parameter for the query.
	 * @return The lookahead parameter specifies how many results are cached 
	 * in the client.  A large value allows quick access to results but may incur 
	 * a delay when the results are transmitted immediately after the query.
	 * A small value implies more round-trips to the server.
	 * <p>
	 * A zero value specifies that all the search results remain on the server.
	 * This option creates a Cursor that can be passed to a serializer in the server.
	 * The initial setting is 1000.
	 */
	public int getLookahead () { return baseInstance.getLookahead(); }
	
	/**
	 * Set the omitNonEncoded option.
	 * When true, non-encoded triples are not included in the search result.
	 * @param v true or false
	 */
	public void setOmitNonEncoded ( boolean v ) {
		freshState();
		baseInstance.setOmitNonEncoded(v);
	}
	
	/**
	 * Query the omitEncoded option.
	 * When true, encoded triples are not included in the search result.
	 * @return 1 if true, 0 if false, -1 if unspecified (ie server default)
	 */
	public int getOmitNonEncoded () {
		return baseInstance.getOmitNonEncoded();
	}
	
	/**
	 * Set the indexedOnly option.
	 * When true, only indexed triples are included in the search result.
	 * @param v true or false
	 */
	public void setIndexedOnly ( boolean v ) {
		freshState();
		baseInstance.setIndexedOnly(v);
	}
	
	/**
	 * Query the indexedOnly option.
	 * When true, only indexed triples are included included in the search result.
	 * @return 1 if true, 0 if false, -1 if unspecified (ie server default)
	 */
	public int getIndexedOnly () {
		return baseInstance.getIndexedOnly();
	}
	
	
	
	
	
	/**
	 * Set the name of a filter function option.
	 * @param name
	 */
	public void setFilterFunction ( String name ) {
		freshState();
		baseInstance.setFilterFunction(name);
	}
	
	/**
	 * Query the name of the filter function.
	 * @return the name of the load-function, or null if none is set.
	 */
	public String getLoadFunction () {
		return baseInstance.getLoadFunction();
	}
	
	
	/**
	 * Query the result of a triples query that returns a collection of triples.
	 * This result is available only after a call to one of the describe()
	 * or construct()
	 * methods.
	 * @return the resultCursor
	 */
	public Cursor getResultCursor() {
		if ( haveResultCursor ) return resultCursor;
		throw new IllegalStateException("ResultCursor is not set.");
	}

	private void validate ( AllegroGraph ag ) {
		freshState();
		if ( ag!=null ) this.ag = ag;
		if ( this.ag==null )
			throw new IllegalStateException("Cannot run without a triple store.");
	}
	
	/**
	 * Specify the triple store agaist which this query will run.
	 * Setting the store clears out any previous results.
	 * @param ag
	 */
	public void setTripleStore ( AllegroGraph ag ) {
		freshState();
		this.ag = ag;
	}
	
	/**
	 * Query the triple store agaist which this query has or will run.
	 * @return the AllegroGraoh instance or null.
	 */
	public AGInner getTripleStore () { return ag; }
	
	
	/**
	 * Count the number of triples located by a pattern of parts.
	 * 
	 * @return the number of results found.  The actual results are discarded.
	 * @throws AllegroGraphException
	 */
	public long count () throws AllegroGraphException {
		freshState();
		return baseInstance.count(ag.agbase);
	}
	
	/**
	 * Count the number of triples located by a pattern of parts.
	 * @param ag The triple store where the query should run.
	 * @return the number of results found.  The actual results are discarded.
	 * @throws AllegroGraphException
	 */
	public long count ( AllegroGraph ag ) throws AllegroGraphException {
		this.ag = ag;
		return count();
	}
	
	
	/**
	 * Estimate the number of triples located by a pattern of parts
	 * using only the information in indices.
	 * Unindexed triples are not included in the estimate.
	 * @param roughly When true, the estimate can be off by as much as twice 
	 *     the triple store's metaindex-skip-size for each index chunk involved.
	 *     When false, return a more acurate (but slower) estimate.
	 * @return the number of results estimated.  The actual results are never located.
	 * @throws AllegroGraphException
	 */
	public long estimate( boolean roughly ) throws AllegroGraphException {
		freshState();
		return baseInstance.estimate(roughly);
	}
	
	public long estimate( boolean roughly, AllegroGraph ag ) throws AllegroGraphException {
		this.ag = ag;
		return baseInstance.estimate(roughly);
	}
	
	/**
	 * Determine if a triple matching a pattern exists..
	 * 
	 * @return true if a triple exists.  The triple is not returned.
	 * @throws AllegroGraphException
	 */
	public boolean ask () throws AllegroGraphException {
		freshState();
		return baseInstance.ask(ag.agbase);
	}
	public boolean ask ( AllegroGraph ag ) throws AllegroGraphException {
		this.ag = ag;
		return ask();
	}
	
	/**
	 * Find one triple that matches a pattern.
	 * Any additional triples are ignored.
	 * @return the Triple instance or null.
	 * @throws AllegroGraphException
	 */
	public Triple find () throws AllegroGraphException {
		freshState();
		com.franz.agbase.Triple tr3 = baseInstance.find();
		if ( tr3==null ) return null;
		resultTriple = AGFactory.makeTriple(ag, tr3);
		return resultTriple;
	}
	
	
	
	/**
	 * Find all the triples that match the pattern.
	 * 
	 * @return A Cursor instance that can iterate over the results.
	 * @throws AllegroGraphException if a problem was encountered during the search.
	 * @throws IllegalArgumentException if this instance is not properly initialized.
	 */
	public Cursor run () throws AllegroGraphException {
		validate(null);
		Object v = baseInstance.run(ag.agbase);
		resultCursor = AGFactory.makeCursor(ag, (TriplesIterator) v);
		return resultCursor;
	}
	public Cursor run ( AllegroGraph ag ) throws AllegroGraphException {
		this.ag = ag;
		return run();
	}
	
	
	/**
	 * Query the hasValue option.
	 * When true, hasValue reasoning is enablede.
	 * @return 1 if true, 0 if false, -1 if unspecified (ie server default)
	 */
	public int getHasValue () { 
		return baseInstance.getHasValue();
		}
	
	/**
	 * Enable or disable hasValue reasoning.
	 * This option is in effect only if reasoning is ebabled.
	 * @param v true or false
	 */
	public void setHasValue ( boolean v ) {
		baseInstance.setHasValue(v);
	}
	
	
}
