package com.franz.agbase.examples;

import com.franz.agbase.*;


public class AGTemporalLoading {

	/**
	 * Demonstrates loading temporal data using a datatype mapping.
	 * 
	 * @param unused
	 * @throws AllegroGraphException 
	 */
	public static void main(String[] args) throws AllegroGraphException {
		
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}

		// Create a fresh triple store for this example. 
		AllegroGraph ts = ags.renew("temporal", AGPaths.TRIPLE_STORES);

		// Add a datatype mapping so that XMLSchema#dateTime typed literals are 
		// loaded as date-time UPI's, enabling more efficient storage and range 
		// querying over dates.
		String[] map = new String[3];
		map[0] = "http://www.w3.org/2000/01/XMLSchema#dateTime";
		map[1] = "date-time";
		map[2] = "datatype";
		ts.addDataMapping(map);
		
		// Load some temporal data
		AGLoadNtriples.loadNTriplesWithTiming(ts, AGPaths.dataSources("temporal.nt"));

		// Index the store
		AGIndexAllTriples.indexAllTriplesWithTiming(ts);
		
		// Register any namespaces
		ts.registerNamespace("ex", "http://example.org/");
		ts.registerNamespace("t", "http://franz.com/ns/allegrograph/3.0/temporal/");

		// Set up a Prolog query object
		PrologSelectQuery q = new PrologSelectQuery();
		q.setTripleStore(ts);
		q.setDistinct(true);

		// Query the store for points and their date-times
		String pquery = "(q ?pt !t:time ?dt)";
		q.setVariables(new String[]{"pt","dt"});
		q.setQuery(pquery);
		ValueSetIterator it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for intervals and their startpoints
		pquery = "(q ?i !t:startpoint ?pt)";
		q.setVariables(new String[]{"i","pt"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for intervals and their endpoints
		pquery = "(q ?i !t:endpoint ?pt)";
		q.setVariables(new String[]{"i","pt"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for intervals and their starttimes
		pquery = "(q ?i !t:starttime ?dt)";
		q.setVariables(new String[]{"i","dt"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for intervals and their endtimes
		pquery = "(q ?i !t:endtime ?dt)";
		q.setVariables(new String[]{"i","dt"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Close the triple store and disconnect from the server.
		ts.closeTripleStore();
		ags.disable();
	}
}
