package com.franz.agbase.examples;

import com.franz.agbase.*;

public class AGGeospatialBoundingBoxSpherical {

	/**
	 * Demonstrates retrieval within a bounding box in a Spherical system.
	 * 
	 * @param args unused
	 * @throws AllegroGraphException
	 */
	public static void main(String[] args) throws AllegroGraphException {
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}

		// Create a fresh triple-store.
		AllegroGraph ts = ags.renew("geospatialboundingbox", AGPaths.TRIPLE_STORES);
		
		// Register namespaces
		ts.registerNamespace("ex","http://example.org/");
		
		// Get a GeoExtension instance for this store to work with geospatial features 
		GeoExtension geo = ts.getGeoExtension();
		
		// Define a geospatial subtype to encode coordinates 
		GeospatialSubtype subtype = geo.registerLatitudeStripingInMiles(50.0);
		geo.addSubtype(subtype);
		
		// Encode some Spherical coordinates as Geospatial UPIs
		// Note the argument ordering is subtype, longitude, latitude
		UPI upiA = geo.encodeUPI(subtype, -122, 37);
		UPI upiB = geo.encodeUPI(subtype, -121, 37);
		UPI upiC = geo.encodeUPI(subtype, -120, 37);
		UPI upiD = geo.encodeUPI(subtype, -121, 36);
		UPI upiE = geo.encodeUPI(subtype, -121, 38);

		// Add triples to the store
		ts.addStatement("!ex:pointA", "!ex:location", upiA);
		ts.addStatement("!ex:pointB", "!ex:location", upiB);
		ts.addStatement("!ex:pointC", "!ex:location", upiC);
		ts.addStatement("!ex:pointD", "!ex:location", upiD);
		ts.addStatement("!ex:pointE", "!ex:location", upiE);
		
		// Index the store
		AGIndexAllTriples.indexAllTriplesWithTiming(ts);
		
		// Show all triples in the store
		TriplesIterator cc = ts.getStatements(null,null,null);
		AGUtils.showTriples("All triples:",cc);
		
		// Show triples within a bounding box
		cc = geo.getStatementsInBoundingBox(subtype, "!ex:location", -122, -120, 36, 38, false, false);
		AGUtils.showTriples("Triples in bounding box:",cc);
		
		// Close the store and disconnect from the server.
		ts.closeTripleStore();
		ags.disable();
	}

}
