package com.franz.agbase.examples;

import com.franz.agbase.*;

public class AGSparqlRDFLiterals {

	/**
	 * Demonstrates basic of matching RDF literals in SPARQL
	 * 
	 * @param args unused
	 * @throws AllegroGraphException 
	 */
	public static void main(String[] args) throws AllegroGraphException {
		
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}
		
		// Create fresh triple-store for this example.
		AllegroGraph ts = ags.renew("sparql", AGPaths.TRIPLE_STORES);
				
		// Register any namespaces
		ts.registerNamespace("dt", "http://example.org/datatype#");
		ts.registerNamespace("ns", "http://example.org/ns#");
		ts.registerNamespace("xsd", "http://www.w3.org/2001/XMLSchema#");
		
		// Add some data to the store
		ts.addStatement("!ns:a","!ns:p", "!\"42\"^^xsd:integer");
		URINode u = (URINode)ts.addPart("!xsd:integer");
		AGUtils.showURI(u);
		ts.addStatement("!ns:b","!ns:p",ts.createLiteral("42", u));
		ts.addStatement("!ns:c","!ns:p",ts.createLiteral("42", ts.createURI("http://www.w3.org/2001/XMLSchema#integer")));
		ts.addStatement("!ns:d","!ns:p",ts.createLiteral("42", ts.createURI("http://www.w3.org/2001/XMLSchema#int")));

		ts.addStatement("!ns:x","!ns:p",ts.createLiteral("cat", "en"));
		URINode v = (URINode)ts.addPart("!dt:specialDatatype");
		AGUtils.showURI(v);
		ts.addStatement("!ns:z","!ns:p",ts.createLiteral("abc",v));
		ts.addStatement("!ns:z2","!ns:p", "!\"abc\"^^dt:specialDatatype");

		// This query has no solution because "cat" is not the same 
		// RDF literal as "cat"@en
		String query =
		"SELECT ?v WHERE { ?v ?p \"cat\" }";
		
		// Query the store and show the results
		SPARQLQuery sq = new SPARQLQuery();
		sq.setTripleStore(ts);
		sq.setQuery(query);
		AGSparqlSelect.doSparqlSelect(sq);
		
		// Now with the language tag "cat"@en
		query =
		"SELECT ?v WHERE { ?v ?p \"cat\"@en }";
		
		// Query the store and show the results
		sq.setQuery(query);
		AGSparqlSelect.doSparqlSelect(sq);
		
		// Now with single quotes instead of escaped double quotes
		query =
		"SELECT ?v WHERE { ?v ?p 'cat'@en }";
		
		// Query the store and show the results
		sq.setQuery(query);
		AGSparqlSelect.doSparqlSelect(sq);
		
		// Integers in a SPARQL query indicate an RDF typed literal
		// with the datatype xsd:integer.  Note that this will not
		// match literals of another datatype, e.g. xsd:int.
		query =
		"SELECT ?v WHERE { ?v ?p 42 }";
		
		// Query the store and show the results
		sq.setQuery(query);
		AGSparqlSelect.doSparqlSelect(sq);
		
		// The query processor does not have to have any understanding 
		// of the values in the space of the datatype. Because the 
		// lexical form and datatype IRI both match, the literal matches.
		query = 
		"SELECT ?v WHERE { ?v ?p 'abc'^^<http://example.org/datatype#specialDatatype> }";
		
		// Query the store and show the results
		sq.setQuery(query);
		AGSparqlSelect.doSparqlSelect(sq);
		
		// Close the triple store and disconnect from the server.
		ts.closeTripleStore();
		ags.disable();
	}

}