package com.franz.agbase.examples;

import com.franz.agbase.*;


public class AGTemporalIntervalIntervalRelations {

	/**
	 * Demonstrates temporal queries involving Allen relations between intervals.
	 * 
	 * @param unused
	 * @throws AllegroGraphException 
	 */
	public static void main(String[] args) throws AllegroGraphException {
		
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}

		// Create a fresh triple store for this example. 
		AllegroGraph ts = ags.renew("temporal", AGPaths.TRIPLE_STORES);

		// Add a datatype mapping so that XMLSchema#dateTime typed literals are 
		// loaded as date-time UPI's, enabling more efficient storage and range 
		// querying over dates.
		String[] map = new String[3];
		map[0] = "http://www.w3.org/2000/01/XMLSchema#dateTime";
		map[1] = "date-time";
		map[2] = "datatype";
		ts.addDataMapping(map);
		
		// Load some temporal data
		AGLoadNtriples.loadNTriplesWithTiming(ts, AGPaths.dataSources("temporal.nt"));

		// Index the store for faster querying
		AGIndexAllTriples.indexAllTriplesWithTiming(ts);
		
		// Register any namespaces
		ts.registerNamespace("ex", "http://example.org/");
		ts.registerNamespace("t", "http://franz.com/ns/allegrograph/3.0/temporal/");

		TriplesIterator cc = ts.getStatements(null, null, null);
		AGUtils.showTriples(cc);
		
		// Set up a Prolog query object
		PrologSelectQuery q = new PrologSelectQuery();
		q.setTripleStore(ts);
		q.setDistinct(true);
		
		// Query the store for intervals satisfying interval-before
		String pquery = "(interval-before ?i !ex:i68)";
		q.setVariables(new String[]{"i"});
		q.setQuery(pquery);
		ValueSetIterator it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for intervals satisfying interval-meets
		pquery = "(interval-meets ?i !ex:i68)";
		q.setVariables(new String[]{"i"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for intervals satisfying interval-overlaps
		pquery = "(interval-overlaps ?i !ex:i38)";
		q.setVariables(new String[]{"i"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for intervals satisfying interval-starts
		pquery = "(interval-starts ?i !ex:i16)";
		q.setVariables(new String[]{"i"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for intervals satisfying interval-during
		pquery = "(interval-during ?i !ex:i27)";
		q.setVariables(new String[]{"i"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for intervals satisfying interval-finishes
		pquery = "(interval-finishes ?i !ex:i38)";
		q.setVariables(new String[]{"i"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for intervals satisfying interval-cotemporal
		pquery = "(interval-cotemporal ?i !ex:i36)";
		q.setVariables(new String[]{"i"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for intervals satisfying interval-after
		pquery = "(interval-after ?i !ex:i45)";
		q.setVariables(new String[]{"i"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for intervals satisfying interval-met-by
		pquery = "(interval-met-by ?i !ex:i14)";
		q.setVariables(new String[]{"i"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for intervals satisfying interval-overlapped-by
		pquery = "(interval-overlapped-by ?i !ex:i36)";
		q.setVariables(new String[]{"i"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for intervals satisfying interval-started-by
		pquery = "(interval-started-by ?i !ex:i14)";
		q.setVariables(new String[]{"i"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for intervals satisfying interval-contains
		pquery = "(interval-contains ?i !ex:i36)";
		q.setVariables(new String[]{"i"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for intervals satisfying interval-finished-by
		pquery = "(interval-finished-by ?i !ex:i48)";
		q.setVariables(new String[]{"i"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Close the triple store and disconnect from the server.
		ts.closeTripleStore();
		ags.disable();
	}

}
