package com.franz.agbase.examples;

import com.franz.agbase.*;


public class AGTemporalPointIntervalRelations {

	/**
	 * Demonstrates temporal queries involving Allen relations between 
	 * points and intervals.
	 * 
	 * @param unused
	 * @throws AllegroGraphException 
	 */
	public static void main(String[] args) throws AllegroGraphException {
		
		// Connect to server, which must already be running.
		AllegroGraphConnection ags = new AllegroGraphConnection();
		try {
			ags.enable();
		} catch (Exception e) {
			throw new AllegroGraphException("Server connection problem", e);
		}

		// Create a fresh triple store for this example. 
		AllegroGraph ts = ags.renew("temporal", AGPaths.TRIPLE_STORES);

		// Add a datatype mapping so that XMLSchema#dateTime typed literals are 
		// loaded as date-time UPI's, enabling more efficient storage and range 
		// querying over dates.
		String[] map = new String[3];
		map[0] = "http://www.w3.org/2000/01/XMLSchema#dateTime";
		map[1] = "date-time";
		map[2] = "datatype";
		ts.addDataMapping(map);
		
		// Load some temporal data
		AGLoadNtriples.loadNTriplesWithTiming(ts, AGPaths.dataSources("temporal.nt"));

		// Index the store for faster querying
		AGIndexAllTriples.indexAllTriplesWithTiming(ts);
		
		// Register any namespaces
		ts.registerNamespace("ex", "http://example.org/");
		ts.registerNamespace("t", "http://franz.com/ns/allegrograph/3.0/temporal/");

		// Set up a Prolog query object
		PrologSelectQuery q = new PrologSelectQuery();
		q.setTripleStore(ts);
		q.setDistinct(true);

		// Query the store for points that are before interval i36
		String pquery = "(point-before-interval ?pt !ex:i36)";
		q.setVariables(new String[]{"pt"});
		q.setQuery(pquery);
		ValueSetIterator it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for intervals starting at pt3
		pquery = "(point-starts-interval !ex:pt3 ?i)";
		q.setVariables(new String[]{"i"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for points during interval i36
		pquery = "(point-during-interval ?pt !ex:i36)";
		q.setVariables(new String[]{"pt"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for intervals ending at point pt6
		pquery = "(point-ends-interval !ex:pt6 ?i)";
		q.setVariables(new String[]{"i"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Query the store for points that are after interval i36
		pquery = "(point-after-interval ?pt !ex:i36)";
		q.setVariables(new String[]{"pt"});
		q.setQuery(pquery);
		it = q.run();
		System.out.println("Results for " + pquery);
		AGUtils.showResults(it);
		
		// Close the triple store and disconnect from the server.
		ts.closeTripleStore();
		ags.disable();
	}

}
