package com.franz.agbase.examples;

import com.franz.agbase.*;

/**
 * Demonstrates some basics of using TriplesIterators and ValueSetIterators
 * and showing results.
 * 
 * This is a general utility class with no main method.
 * 
 */
public class AGUtils {
	
	/**
	 * A default limit on the number of triples to show, to avoid showing
	 * overwhelmingly many. 
	 */
	private static int showLimit = 100;
	
	public static void showTriples(String msg, TriplesIterator it) throws AllegroGraphException {
		System.out.println(msg);
		showTriples(it);
	}
	
	public static void showTriples(TriplesIterator it) throws AllegroGraphException {
		showTriples(it, showLimit);
	}
	
	public static void showTriples(TriplesIterator it, int limit) throws AllegroGraphException {
		int i=0;
		while (i<limit && it.step()) {
			showTriple(it.getTriple());
			i++;
		}
		if (it.step()) {
			System.out.println("[Showing " + limit + " triples, there are more]");
		}
	}
	
	public static void showTriple(Triple tr) throws AllegroGraphException {
		System.out.println(tr.toString());
	}

	public static void showResults(ValueSetIterator it) throws AllegroGraphException {
		String[] var = it.getNames();
	    System.out.println("Number of solutions: " + it.getCount());
	    for (int i=0; it.hasNext(); i++) {
			ValueObject[] objects = it.next();
			System.out.println("Solution " + (i+1) + ":");
			for (int j = 0; j < objects.length; j++) {
				ValueObject term = objects[j];
				System.out.println("  " + var[j] + " = " + printValueObject(term));
			}
		}
	}

	public static String printValueObject(ValueObject o) {
		String result;
		if (o == null) {
			result = "Null";
		} else if (o instanceof LiteralNode) {
			LiteralNode l = (LiteralNode)o;
			result = l.getLabel();
		} else if (o instanceof BlankNode) {
			BlankNode b = (BlankNode)o;
			result = b.getID();
		} else if (o instanceof ValueNode) {
			ValueNode n = (ValueNode)o;
			result = n.toString();
		} else {
			result = o.toString();
		}
		return result;
	}

	public static void showURI (URINode r) {
		System.out.println(r.getURI() + ":");
		System.out.println("  Namespace: " + r.getNamespace());
		System.out.println("  LocalName: " + r.getLocalName());
	}

	public static void showLiteral (LiteralNode lit) {
		System.out.println("Literal: " + lit.toString());
		System.out.println("  Label: " + lit.getLabel());
		System.out.println("  Datatype: " + lit.getDatatype());
		System.out.println("  Language: " + lit.getLanguage());
		// Note that only Literals added to the store have a UPI
		System.out.println("  UPI: " + ((ValueNode)lit).queryAGId());
	}
	
	public static void showEncodedLiteral (EncodedLiteral lit) {
		System.out.println("EncodedLiteral: " + lit.toString());
		System.out.println("  Label: " + lit.getLabel());
		System.out.println("  Datatype: " + lit.getDatatype());
		System.out.println("  Language: " + lit.getLanguage());
		// Note that only Literals added to the store have a UPI
		System.out.println("  UPI: " + ((ValueNode)lit).queryAGId());
	}
	
	public static void printObjectArray(String msg, Object[] objArr) {
		System.out.println(msg);
		if (objArr != null) {
			for (int i=0; i<objArr.length;i++) {
				System.out.println("  "+i+": "+objArr[i]);
			}
		}
	}
	
	public static void printUPIArray(String msg, AllegroGraph ts, UPI[] objArr) throws AllegroGraphException {
		System.out.println(msg);
		if (objArr != null) {
			for (int i=0; i<objArr.length;i++) {
				String[] parts = ts.getParts(objArr[i]);
				System.out.println("  "+i+": "+parts[1]);
			}
		}
	}
	
	public static String upiArrayToString(AllegroGraph ts, UPI[] objArr) throws AllegroGraphException {
		StringBuffer buf = new StringBuffer();
		if (objArr != null) {
			buf.append("{");
			for (int i=0; i<objArr.length;i++) {
				String[] parts = ts.getParts(objArr[i]);
				buf.append(parts[1]).append(" ");
			}
			buf.append("}");
		}
		return buf.toString();
	}
	
	public static void printStringArray(String msg, String[] objArr) {
		System.out.println(msg);
		if (objArr != null) {
			for (int i=0; i<objArr.length;i++) {
				System.out.println("  "+i+": "+objArr[i]);
			}
		}
	}
	
	public static String elapsedTime(long start) {
		long total = System.currentTimeMillis() - start;
		long min = total/60000;
		long msec = total%60000;
		double sec = msec/1000.0;
		String report;
		if (min > 0) {
			report = min + ":" + sec + " minutes:seconds";
		} else {
			report = sec + " seconds";
		}
		return report;
	}
	
}